/* -*-pgsql-c-*- */
/*
 * $Header: /home/t-ishii/repository/pgpool/child.c,v 1.22 2004/04/28 13:52:38 t-ishii Exp $
 *
 * pgpool: a language independent connection pool server for PostgreSQL 
 * written by Tatsuo Ishii
 *
 * Copyright (c) 2003, 2004	Tatsuo Ishii
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice appear in all
 * copies and that both that copyright notice and this permission
 * notice appear in supporting documentation, and that the name of the
 * author not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior
 * permission. The author makes no representations about the
 * suitability of this software for any purpose.  It is provided "as
 * is" without express or implied warranty.
 *
 * child.c: child process main
 *
 */
#include "config.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif

#include <signal.h>

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#include "pool.h"

#ifdef NONE_BLOCK
static void set_nonblock(int fd);
#endif
static void unset_nonblock(int fd);

static POOL_CONNECTION *do_accept(int unix_fd, int inet_fd);
static StartupPacket *read_startup_packet(POOL_CONNECTION *cp);
static int send_startup_packet(POOL_CONNECTION *cp, StartupPacket *sp);
static POOL_CONNECTION_POOL *connect_backend(StartupPacket *sp, POOL_CONNECTION *frontend);
static void reset_backend(POOL_CONNECTION *backend);
static void cancel_request(CancelPacket *cp, int secondary_backend);
static RETSIGTYPE die(int sig);

/*
* child main loop
*/
void do_child(int unix_fd, int inet_fd)
{
	StartupPacket *sp;
	POOL_CONNECTION *frontend;
	POOL_CONNECTION_POOL *backend;

	pool_debug("I am %d", getpid());
	/* set up signal handlers */
	signal(SIGALRM, SIG_DFL);
	signal(SIGTERM, die);
	signal(SIGINT, SIG_DFL);
	signal(SIGCHLD, SIG_DFL);
	signal(SIGUSR1, SIG_DFL);
	signal(SIGPIPE, SIG_IGN);

#ifdef NONE_BLOCK
	/* set listen fds to none block */
	set_nonblock(unix_fd);
	if (inet_fd)
	{
		set_nonblock(inet_fd);
	}
#endif

	/* initialize connection pool */
	if (pool_init_cp())
	{
		exit(1);
	}

	for (;;)
	{
		int major;
		int connection_reuse = 1;
		int ssl_request = 0;

		/* perform accept() */
		frontend = do_accept(unix_fd, inet_fd);
		if (frontend == NULL)
		{
			/* accept() failed. return to the accept() loop */
			continue;
		}

		/* set frontend fd to blocking */
		unset_nonblock(frontend->fd);

		/* disable timeout */
		pool_disable_timeout();

		/* read the startup packet */
	retry_startup:
		sp = read_startup_packet(frontend);
		if (sp == NULL)
		{
			/* failed to read the startup packet. return to the
			   accept() loop */
			pool_close(frontend);
			continue;
		}

		major = ntohl(sp->protoVersion) >> 16;
		pool_debug("major: %d", major);

		/* V3(7.4 or later) protocol? */
		if (major < 1234 && major > 2)
		{
			static char *msg = "Sorry, we do not support V3 protocol";

			pool_debug("V3 protocol request received. attempt to fallback to V2");

			/*
			 * Send "we do not suppport V3 protocol" error in the hope that
			 * it could fallback to V2 protocol..,.
			 */
			pool_write_and_flush(frontend, "E", 1);
			pool_write_and_flush(frontend, msg, strlen(msg)+1);
			pool_close(frontend);
			continue;
		}

		/* cancel request? */
		if (ntohl(sp->protoVersion) == 0x04D2162E)		/* 0x04D2162F = 12345678 */
		{
			cancel_request((CancelPacket *)sp, 0);
			if (pool_config.replication_enabled)
				cancel_request((CancelPacket *)sp, 1);
			continue;
		}

		/* SSL? */
		if (ntohl(sp->protoVersion) == 0x04D2162F)		/* 0x04D2162F = 12345679 */
		{
			/* SSL not supported */
			pool_debug("SSLRequest: sent N; retry startup");
			if (ssl_request)
			{
				pool_close(frontend);
				continue;
			}

			/*
			 * say to the frontend "we do not suppport SSL"
			 * note that this is not a NOTICE response despite it's an 'N'!
			 */
			pool_write_and_flush(frontend, "N", 1);
			ssl_request = 1;
			goto retry_startup;
		}

		/*
		 * if there's no connection associated with user and database,
		 * we need to connect to the backend and send the startup packet.
		 */
		if ((backend = pool_get_cp(sp->user, sp->database)) == NULL)
		{
			connection_reuse = 0;

			if ((backend = connect_backend(sp, frontend)) == NULL)
				continue;
		}
		else
		{
			reset_backend(MASTER_CONNECTION(backend)->con);
			if (pool_config.replication_enabled)
				reset_backend(SECONDARY_CONNECTION(backend)->con);

			/* send Auth OK and Ready for Query to the frontend */
			if (pool_send_auth_ok(frontend, MASTER_CONNECTION(backend)->pid, MASTER_CONNECTION(backend)->key) != POOL_CONTINUE)
			{
				pool_close(frontend);
				continue;
			}
			if (pool_write_and_flush(frontend, "Z", 1) < 0)
			{
				pool_close(frontend);
				continue;
			}
		}

		/* enable query result read timeout if non strict mode */
		if (pool_config.replication_strict == 0)
			pool_enable_timeout();

		/* query process loop */
		for (;;)
		{
			int status;

			status = pool_process_query(frontend, backend, connection_reuse);
			switch (status)
			{
				/* client exits */
				case POOL_END:
					pool_close(frontend);

					/* do not cache connection to template0, template1, regression */
					if (!strcmp(sp->database, "template0") || !strcmp(sp->database, "template1") ||
						!strcmp(sp->database, "regression"))
						pool_discard_cp(sp->user, sp->database);
					else
						pool_connection_pool_timer(backend);
					break;
				
				/* error occured. discard backend connection pool
                   and disconnect connection to the frontend */
				case POOL_ERROR:
#ifdef NOT_USED
					pool_discard_cp(sp->user, sp->database);
					pool_close(frontend);
#endif
					notice_backend_error();
					exit(1);
					break;

				/* fatal error occured. just exit myself... */
				case POOL_FATAL:
					notice_backend_error();
					exit(1);
					break;

				/* not implemented yet */
				case POOL_IDLE:
					do_accept(unix_fd, inet_fd);
					pool_debug("accept while idle");
					break;
			}

			if (status != POOL_CONTINUE)
				break;
		}

	}
	exit(0);
}

/* -------------------------------------------------------------------
 * private functions
 * -------------------------------------------------------------------
 */

#ifdef NONE_BLOCK
/*
 * set non-block flag
 */
static void set_nonblock(int fd)
{
	int var;

	/* set fd to none blocking */
	var = fcntl(fd, F_GETFL, 0);
	if (var == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
	if (fcntl(fd, F_SETFL, var | O_NONBLOCK) == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
}
#endif

/*
 * unset non-block flag
 */
static void unset_nonblock(int fd)
{
	int var;

	/* set fd to none blocking */
	var = fcntl(fd, F_GETFL, 0);
	if (var == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
	if (fcntl(fd, F_SETFL, var & ~O_NONBLOCK) == -1)
	{
		pool_error("fcntl failed. %s", strerror(errno));
		exit(1);
	}
}

/*
* perform accept() and returns new fd
*/
static POOL_CONNECTION *do_accept(int unix_fd, int inet_fd)
{
    fd_set	readmask;
    int fds;

	struct sockaddr addr;
	socklen_t addrlen;
	int fd = 0;
	int afd;
	int inet = 0;
	POOL_CONNECTION *cp;

	FD_ZERO(&readmask);
	FD_SET(unix_fd, &readmask);
	if (inet_fd)
		FD_SET(inet_fd, &readmask);

	fds = select(Max(unix_fd, inet_fd)+1, &readmask, NULL, NULL, NULL);
	if (fds == -1)
	{
		if (errno == EAGAIN || errno == EINTR)
			return NULL;

		pool_error("select() failed. reason %s", strerror(errno));
		return NULL;
	}

	if (fds == 0)
		return NULL;

	if (FD_ISSET(unix_fd, &readmask))
	{
		fd = unix_fd;
	}

	if (FD_ISSET(inet_fd, &readmask))
	{
		fd = inet_fd;
		inet++;
	}

	/*
	 * Note that some SysV systems do not work here. For those
	 * systems, we need some locking mechanism for the fd.
	 */
	addrlen = sizeof(addr);
	afd = accept(fd, &addr, &addrlen);
	if (afd < 0)
	{
		pool_error("accept() failed. reason: %s", strerror(errno));
		return NULL;
	}
	pool_debug("I am %d accept fd %d", getpid(), afd);

	/* set NODELAY and KEEPALIVE options if INET connection */
	if (inet)
	{
		int on = 1;

		if (setsockopt(afd, IPPROTO_TCP, TCP_NODELAY,
					   (char *) &on,
					   sizeof(on)) < 0)
		{
			pool_error("do_accept: setsockopt() failed: %s", strerror(errno));
			close(afd);
			return NULL;
		}
		if (setsockopt(afd, SOL_SOCKET, SO_KEEPALIVE,
					   (char *) &on,
					   sizeof(on)) < 0)
		{
			pool_error("do_accept: setsockopt() failed: %s", strerror(errno));
			close(afd);
			return NULL;
		}
	}

	if ((cp = pool_open(afd)) == NULL)
	{
		close(afd);
		return NULL;
	}
	return cp;
}

/*
* read startup packet
*/
static StartupPacket *read_startup_packet(POOL_CONNECTION *cp)
{
	static StartupPacket sp;
	int len;

	memset((char *)&sp, 0, sizeof(sp));

	if (pool_read(cp, &len, sizeof(len)))
	{
		return NULL;
	}
	len = ntohl(len);
	len -= sizeof(len);

	if (pool_read(cp, &sp, len))
	{
		return NULL;
	}

	pool_debug("Protocol Version: %08x",ntohl(sp.protoVersion));
	pool_debug("Protocol Major: %d Minor:%d",
		   ntohl(sp.protoVersion)>>16,
			   ntohl(sp.protoVersion) & 0x0000ffff);
	pool_debug("database: %s",sp.database);
	pool_debug("user: %s",sp.user);

	return &sp;
}

/*
* send startup packet
*/
static int send_startup_packet(POOL_CONNECTION *cp, StartupPacket *sp)
{
	int len;

	len = htonl(sizeof(len) + sizeof(StartupPacket));

	pool_write(cp, &len, sizeof(len));
	return pool_write_and_flush(cp, sp, sizeof(StartupPacket));
}

/*
 * reset backend status
 */
static void reset_backend(POOL_CONNECTION *backend)
{
	char kind;
	int len;

#ifdef NO_RESET_ALL
	static char *queries[] = {"ABORT"};
#else
	static char *queries[] = {"ABORT", "RESET ALL"};
#endif

	int i;

	for (i=0;i<sizeof(queries)/sizeof(char *);i++)
	{
		pool_write(backend, "Q", 1);
		pool_write_and_flush(backend, queries[i], strlen(queries[i])+1);
		pool_read(backend, &kind, 1);

		switch(kind)
		{
			case 'C':
			case 'E':
			case 'N':
				/* read and discard notice or error message */
				pool_read_string(backend, &len, 0);

				if (kind != 'C')
				{
					/* read and discard complete command response */
					pool_read(backend, &kind, 1);
					pool_read_string(backend, &len, 0);
				}

				/* read and discard ready for query message */
				pool_read(backend, &kind, 1);
				break;

			default:
				pool_error("reset_backend: Unknown response");
				exit(1);
		}
	}
}

static void cancel_request(CancelPacket *cp, int secondary_backend)
{
	int	len;
	int fd;
	POOL_CONNECTION *con;

	pool_debug("Cancel request received");

	if (secondary_backend)
	{
		if (*pool_config.secondary_backend_host_name == '\0')
			fd = connect_unix_domain_socket(1);
		else
			fd = connect_inet_domain_socket(1);
	}
	else
	{
		if (*pool_config.current_backend_host_name == '\0')
			fd = connect_unix_domain_socket(0);
		else
			fd = connect_inet_domain_socket(0);
	}

	if (fd < 0)
	{
		pool_error("Could not create socket for sending cancel request");
		return;
	}

	con = pool_open(fd);
	if (con == NULL)
		return;

	len = htonl(sizeof(len) + sizeof(CancelPacket));
	pool_write(con, &len, sizeof(len));

	if (pool_write_and_flush(con, cp, sizeof(CancelPacket)) < 0)
		pool_error("Could not send cancel request packet");
	pool_close(con);
}

static POOL_CONNECTION_POOL *connect_backend(StartupPacket *sp, POOL_CONNECTION *frontend)
{
	POOL_CONNECTION_POOL *backend;

	/* connect to the backend and send the startup packect here */
	backend = pool_create_cp(sp->user, sp->database);
	if (backend == NULL)
	{
		static char *msg = "Sorry, too many clients already";

		pool_error("do_child: cannot not create backend connection");

		pool_write(frontend, "E", 1);
		pool_write_and_flush(frontend, msg, strlen(msg)+1);
		pool_close(frontend);
		return NULL;
	}

	/* mark this is a backend connection */
	backend->slots[0]->con->isbackend = 1;
	if (pool_config.replication_enabled)
	{
		backend->slots[1]->con->isbackend = 1;
		backend->slots[1]->con->issecondary_backend = 1;
	}

	/* send startup packet */
	if (send_startup_packet(backend->slots[0]->con, sp) < 0)
	{
		pool_error("do_child: fails to send startup packet to the backend");
		pool_close(frontend);
		return NULL;
	}

	/* send startup packet */
	if (pool_config.replication_enabled)
	{
		if (send_startup_packet(backend->slots[1]->con, sp) < 0)
		{
			pool_error("do_child: fails to send startup packet to the secondary backend");
			pool_close(frontend);
			return NULL;
		}
	}

	/*
	 * do authentication stuff
	 */
	if (pool_do_auth(frontend, backend))
	{
		pool_close(frontend);
		pool_discard_cp(sp->user, sp->database);
		return NULL;
	}
	return backend;
}

static RETSIGTYPE die(int sig)
{
	exit(0);
}
