/*-------------------------------------------------------------------------
 *
 * plruntime.c				- Runtime for the PL/pgPSM procedural language
 *
 * Portions Copyright (c) 1996-2012, PostgreSQL Global Development Group
 * Portions Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *	  src/pl/plpgpsm/src/plruntime.c
 *
 *-------------------------------------------------------------------------
 */

#include "plpgpsm.h"

#include "catalog/pg_type.h"
#include "executor/spi.h"
#include "parser/parse_type.h"
#include "utils/datum.h"
#include "utils/lsyscache.h"
#include "utils/syscache.h"
#include "utils/typcache.h"

#include "utils/builtins.h"

static ParamListInfo setup_param_list(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql);
static void param_fetch(ParamListInfo params, int paramid);

/*
 * create exec state for current function
 *
 */
PLpgPSM_execstate *
plpgpsm_init_execstate(PLpgPSM_execstate *estate, PLpgPSM_function *func, FunctionCallInfo fcinfo)
{
	int	i;

	estate->func = func;
	estate->fcinfo = fcinfo;

	estate->rtinfo = NULL;
	estate->curr_stmt = NULL;
	estate->curr_esql = NULL;
	estate->stop_node = NULL;
	estate->used_tinfos = NIL;
	estate->eval_tuptable = NULL;
	estate->eval_processed = 0;
	estate->ndatums = func->parser->nvars;
	estate->datums = (PLpgPSM_datum *) palloc0(estate->ndatums * sizeof(PLpgPSM_datum));

	/* all variables are null initially */
	for (i = 0; i < estate->ndatums; i++)
		estate->datums[i].isnull = true;

	estate->exec_cxt = CurrentMemoryContext;

	estate->edata = NULL;
	estate->stacked_edata = NULL;

	estate->handler = NULL;

	return estate;
}

void
plpgpsm_clean_execstate(PLpgPSM_execstate *estate)
{
	ListCell *lc;

	plpgpsm_eval_cleanup(estate);

	foreach (lc, estate->used_tinfos)
	{
		pfree(lfirst(lc));
	}

	pfree(estate->datums);
}

/*
 * Release temporary memory used for expression evaluation
 *
 */
void
plpgpsm_eval_cleanup(PLpgPSM_execstate *estate)
{
	if (estate->eval_tuptable != NULL)
	{
		SPI_freetuptable(estate->eval_tuptable);
		estate->eval_tuptable = NULL;
	}

	estate->curr_esql = NULL;
}

/*
 * set a datum. It is used for initial parameter's variables setting
 *
 */
void
plpgpsm_set_datum(PLpgPSM_execstate *estate, PLpgPSM_datum *datum, Datum value, bool isnull, bool copy_datum)
{
	PLpgPSM_typeinfo	*tinfo = datum->typeinfo;

	/* relase memory when it is necessary */
	if (!datum->isnull)
	{
		Assert(tinfo != NULL);

		if (!tinfo->typbyval)
			pfree(DatumGetPointer(datum->value));
	}

	if (!isnull)
	{
		Assert(tinfo != NULL);

		if (copy_datum)
			datum->value = datumCopy(value, tinfo->typbyval, tinfo->typlen);
		else
			datum->value = value;

		datum->isnull = false;
	}
	else
	{
		datum->value = (Datum) 0;
		datum->isnull = true;
	}
}

/*
 * store result of evaluated expression to variable
 *
 *    does IO cast when it is necessary (ToDo - do functional cast instead)
 */
void
plpgpsm_store_result(PLpgPSM_execstate *estate, int attrn, PLpgPSM_datum *datum)
{
	PLpgPSM_typeinfo *tinfo = datum->typeinfo;
	Oid typeid = SPI_gettypeid(estate->eval_tuptable->tupdesc, attrn);
	int32 typmod = estate->eval_tuptable->tupdesc->attrs[attrn - 1]->atttypmod;
	Datum		value;
	bool		isnull;
	bool			copy_datum = true;

	if (tinfo == NULL)
	{
		/*
		 * In a few situation when we wainting for execution to getting variable type,
		 * see CASE statement. In this case, target variable must be NULL;
		 *
		 */
		Assert(datum->isnull);

		tinfo = plpgpsm_make_typeinfo_typeid(estate, typeid, typmod);
		datum->typeinfo = tinfo;
	}

	value = SPI_getbinval(estate->eval_tuptable->vals[attrn - 1], estate->eval_tuptable->tupdesc, attrn, &isnull);
	if (!isnull)
	{
		if (tinfo->typeid != typeid || tinfo->typmod != typmod)
		{

			bool		typIsVarlena;
			Oid		typoutput;
			FmgrInfo		proc;
			char		*str;

			getTypeOutputInfo(typeid, &typoutput, &typIsVarlena);
			fmgr_info_cxt(typoutput, &proc, CurrentMemoryContext);

			str = OutputFunctionCall(&proc, value);
			value = InputFunctionCall(&tinfo->typinput, str, tinfo->typioparam, tinfo->typmod);

			/* now we have a copy */
			copy_datum = false;
			pfree(str);
		}
	}

	plpgpsm_set_datum(estate, datum, value, isnull, copy_datum);
}

/*
 * Used in assign statement for simultaneous assign
 *
 */
void
plpgpsm_store_composed_result_to_datums(PLpgPSM_execstate *estate, List *datums)
{
	Datum		*recvalues;
	bool		*recnulls;
	Datum value;
	bool	isnull;
	HeapTupleHeader		rec;
	TupleDesc		rectupdesc;
	Oid			rectuptyp;
	int32			rectuptypmod;
	HeapTupleData		rectuple;
	int	ncolumns;
	ListCell		*current_datum = list_head(datums);
	int i;

	value = SPI_getbinval(estate->eval_tuptable->vals[0], estate->eval_tuptable->tupdesc, 1, &isnull);
	if (isnull)
		elog(ERROR, "value is null");

	rec = DatumGetHeapTupleHeader(value);

	/* Extract type info from the tuple itself */
	rectuptyp = HeapTupleHeaderGetTypeId(rec);
	rectuptypmod = HeapTupleHeaderGetTypMod(rec);
	rectupdesc = lookup_rowtype_tupdesc(rectuptyp, rectuptypmod);
	ncolumns = rectupdesc->natts;

	/* Build a temporary HeapTuple control structure */
	rectuple.t_len = HeapTupleHeaderGetDatumLength(rec);
	ItemPointerSetInvalid(&(rectuple.t_self));
	rectuple.t_tableOid = InvalidOid;
	rectuple.t_data = rec;

	recvalues = (Datum *) palloc(ncolumns * sizeof(Datum));
	recnulls = (bool *) palloc(ncolumns * sizeof(bool));
					    
	/* Break down the tuple into fields */
	heap_deform_tuple(&rectuple, rectupdesc, recvalues, recnulls);

	for (i = 0; i < ncolumns; i++)
	{
		Oid	columntyp = rectupdesc->attrs[i]->atttypid;
		int32	columntypmod = rectupdesc->attrs[i]->atttypmod;
		PLpgPSM_datum *datum;
		PLpgPSM_variable *var;
		PLpgPSM_referer *ref;

		/* Ignore dropped columns */
		if (rectupdesc->attrs[i]->attisdropped)
			continue;

		if (current_datum == NULL)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
				errmsg("too few values in ASSIGN statement")));

		ref = (PLpgPSM_referer *) lfirst(current_datum);
		current_datum = lnext(current_datum);

		Assert(ref->astnode.type == PLPGPSM_REFERER);
		if (ref->ref->type != PLPGPSM_VARIABLE)
			continue;

		var = (PLpgPSM_variable *) ref->ref;
		datum = &estate->datums[var->offset];

		if (!recnulls[i])
		{
			PLpgPSM_typeinfo *tinfo = datum->typeinfo;

			if (tinfo->typeid != columntyp || tinfo->typmod != columntypmod)
			{
				char *outstr;
				bool		typIsVarlena;
				Oid		typoutput;
				FmgrInfo		proc;
				Datum		value;

				getTypeOutputInfo(columntyp, &typoutput, &typIsVarlena);
				fmgr_info_cxt(typoutput, &proc, CurrentMemoryContext);
				outstr = OutputFunctionCall(&proc, recvalues[i]);
				value = InputFunctionCall(&tinfo->typinput, outstr, tinfo->typioparam, tinfo->typmod);
				plpgpsm_set_datum(estate, datum, value, false, false);
				pfree(outstr);
			}
			else
			{
				plpgpsm_set_datum(estate, datum, recvalues[i], false, true);
			}
		}
		else
			plpgpsm_set_datum(estate, datum, (Datum) 0, true, false);
	}

	if (current_datum != NULL)
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
			errmsg("too many values in ASSIGN statement")));

	ReleaseTupleDesc(rectupdesc);
}

PLpgPSM_typeinfo *
plpgpsm_make_typeinfo(PLpgPSM_execstate *estate, PLpgPSM_ident *ident)
{
	Oid		typeid;
	int32		typmod;

	Assert (ident->astnode.type == PLPGPSM_IDENT_TYPE);

	/* Let the main parser try to parse it under standard SQL rules */
	parseTypeString(ident->name, &typeid, &typmod);

	return plpgpsm_make_typeinfo_typeid(estate, typeid, typmod);
}

PLpgPSM_typeinfo *
plpgpsm_make_typeinfo_typeid(PLpgPSM_execstate *estate, Oid typeid, int32 typmod)
{
	PLpgPSM_typeinfo	*tinfo;
	HeapTuple		typeTup;
	Form_pg_type		typeStruct;
	ListCell	*lc;

	/* try to reuse existing typeinfo */
	foreach (lc, estate->used_tinfos)
	{
		PLpgPSM_typeinfo *t = (PLpgPSM_typeinfo *) lfirst(lc);

		if (t->typeid == typeid && t->typmod == typmod)
			return t;
	}

	typeTup = SearchSysCache1(TYPEOID, ObjectIdGetDatum(typeid));
	if (!HeapTupleIsValid(typeTup))
		elog(ERROR, "cache lookup failed for type %u", typeid);
	typeStruct = (Form_pg_type) GETSTRUCT(typeTup);

	tinfo = (PLpgPSM_typeinfo *) palloc(sizeof(PLpgPSM_typeinfo));

	tinfo->typename = pstrdup(NameStr(typeStruct->typname));
	tinfo->typeid = typeid;
	tinfo->typmod = typmod;
	tinfo->typbyval = typeStruct->typbyval;
	tinfo->typlen = typeStruct->typlen;
	tinfo->typioparam = getTypeIOParam(typeTup);
	fmgr_info(typeStruct->typinput, &(tinfo->typinput));
	fmgr_info(typeStruct->typoutput, &(tinfo->typoutput));

	estate->used_tinfos = lappend(estate->used_tinfos, tinfo);

	ReleaseSysCache(typeTup);

	return tinfo;
}

/*
 * evaluate expression
 *
 *       expected_typeinfo is not used yet. It can be used for specification of target
 *       type - it can minimalize of necessity of IO cast.
 */
void
plpgpsm_eval_expr(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql,
								PLpgPSM_typeinfo *typeinfo,
								bool recheck_single_value)
{
	StringInfoData		str;
	ParamListInfo paramLI;
	SPIPlanPtr	plan;
	int rc;

	if (esql->curr_estate != (psm_estate) estate)
	{
		if (esql->plan != NULL)
		{
			SPI_freeplan(esql->plan);
			esql->plan = NULL;
		}

		if (esql->var_refs != NIL)
		{
			ListCell	*lc;
			foreach(lc, esql->var_refs)
			{
				pfree(lfirst(lc));
			}

			list_free(esql->var_refs);
			esql->var_refs = NIL;
		}
	}

	if (esql->plan == NULL)
	{
		initStringInfo(&str);

		/*
		 * Brackets are necessary. It is necessary, because we cannot to parse
		 * just expression. Brackets eliminate expressions like 10 x, a FROM tab, ..
		 *
		 */
		appendStringInfo(&str, "SELECT (%s)", esql->sqlstr);

		esql->curr_estate = (psm_estate) estate;
		plan = SPI_prepare_params(str.data, (ParserSetupHook) plpgpsm_parser_setup, (void *) esql, 0);

		if (plan == NULL)
		{
			/* Some SPI errors deserve specific error messages */
			switch (SPI_result)
			{
				case SPI_ERROR_COPY:
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("cannot COPY to/from client in PL/pgPSM")));
				case SPI_ERROR_TRANSACTION:
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("cannot begin/end transactions in PL/pgPSM"),
							 errhint("Use a BEGIN block with an EXCEPTION clause instead.")));
				default:
					elog(ERROR, "SPI_prepare_params failed for \"%s\": %s",
						 esql->sqlstr, SPI_result_code_string(SPI_result));
			}
		}

		SPI_keepplan(plan);
		esql->plan = plan;

		pfree(str.data);
	}

	paramLI = setup_param_list(estate, esql);

	rc = SPI_execute_plan_with_paramlist(esql->plan, paramLI, true, 2);
	if (rc != SPI_OK_SELECT)
		elog(ERROR, "cannot to execute expr");

	estate->eval_tuptable = SPI_tuptable;
	estate->eval_processed = SPI_processed;

	/*
	 * Check that the expression returns exactly one column...
	 */
	if (recheck_single_value)
	{
		if (estate->eval_tuptable->tupdesc->natts != 1)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
					 errmsg_plural("query \"%s\" returned %d column",
								   "query \"%s\" returned %d columns",
								   estate->eval_tuptable->tupdesc->natts,
								   estate->curr_esql->sqlstr,
								   estate->eval_tuptable->tupdesc->natts)));

		if (estate->eval_processed != 1)
			ereport(ERROR,
					(errcode(ERRCODE_CARDINALITY_VIOLATION),
					 errmsg("query \"%s\" returned more than one row",
							estate->curr_esql->sqlstr)));
	}

	pfree(paramLI);

	estate->sqlcode = 0;
	estate->edata = NULL;
}

/*
 * evaluate cstring expression
 *
 */
char *
plpgpsm_eval_cstring(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql)
{
	char	*result;

	plpgpsm_eval_expr(estate, esql, NULL, true);
	result = SPI_getvalue(SPI_tuptable->vals[0], SPI_tuptable->tupdesc, 1);
	plpgpsm_eval_cleanup(estate);

	return result;
}

/*
 * evaluate boolean expression
 *
 */
bool
plpgpsm_eval_boolean(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql)
{
	Oid typeid;
	Datum		value;
	bool		isnull;
	bool	result;

	plpgpsm_eval_expr(estate, esql, NULL, true);

	typeid = SPI_gettypeid(estate->eval_tuptable->tupdesc, 1);

	if (typeid != BOOLOID)
		elog(ERROR, "result of expression \"%s\" is not of boolean type",
					esql->sqlstr);

	value = SPI_getbinval(estate->eval_tuptable->vals[0], estate->eval_tuptable->tupdesc, 1, &isnull);
	if (isnull)
		result = false;
	else
		result = DatumGetBool(value);

	plpgpsm_eval_cleanup(estate);

	return result;
}

/*
 * evaluate expression to target typeinfo
 *
 *      now it is used for 
 *
 */
Datum
plpgpsm_eval_datum(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql, PLpgPSM_typeinfo *tinfo, bool *isnull)
{
	Oid typeid;
	int32 typmod;
	Datum	value;

	plpgpsm_eval_expr(estate, esql, tinfo, true);

	typeid = SPI_gettypeid(estate->eval_tuptable->tupdesc, 1);
	typmod = estate->eval_tuptable->tupdesc->attrs[0]->atttypmod;

	value = SPI_getbinval(estate->eval_tuptable->vals[0], estate->eval_tuptable->tupdesc, 1, isnull);

	if (!*isnull)
	{
		if (tinfo->typeid != typeid || tinfo->typmod != typmod)
		{
			bool		typIsVarlena;
			Oid		typoutput;
			FmgrInfo		proc;
			char		*str;

			getTypeOutputInfo(typeid, &typoutput, &typIsVarlena);
			fmgr_info_cxt(typoutput, &proc, CurrentMemoryContext);

			str = OutputFunctionCall(&proc, value);
			value = InputFunctionCall(&tinfo->typinput, str, tinfo->typioparam, tinfo->typmod);

			/* now we have a copy */
			pfree(str);
		}
		else
			value = datumCopy(value, tinfo->typbyval, tinfo->typlen);
	}

	plpgpsm_eval_cleanup(estate);

	return value;
}

Datum
plpgpsm_cast(PLpgPSM_execstate *estate, Datum value, bool isnull,
								Oid	typeid, int32 typmod,
									    PLpgPSM_typeinfo *tinfo)
{
	if (!isnull)
	{
		if (tinfo->typeid != typeid || tinfo->typmod != typmod)
		{
			bool		typIsVarlena;
			Oid		typoutput;
			FmgrInfo		proc;
			char		*str;

			getTypeOutputInfo(typeid, &typoutput, &typIsVarlena);
			fmgr_info_cxt(typoutput, &proc, CurrentMemoryContext);

			str = OutputFunctionCall(&proc, value);
			value = InputFunctionCall(&tinfo->typinput, str, tinfo->typioparam, tinfo->typmod);

			/* now we have a copy */
			pfree(str);
		}
		else
			return value;
	}

	return (Datum) 0;
}

/*
 * prepare relation between plan and estate parameters
 *
 */
static ParamListInfo
setup_param_list(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql)
{
	ParamListInfo  paramLI;

	int	nparams = list_length(esql->var_refs);

	paramLI = (ParamListInfo)
			palloc0(offsetof(ParamListInfoData, params) +
					nparams * sizeof(ParamExternData));
	paramLI->paramFetch = param_fetch;
	paramLI->paramFetchArg = (void *) estate;
	paramLI->parserSetup = (ParserSetupHook) plpgpsm_parser_setup;
	paramLI->parserSetupArg = (void *) esql;
	paramLI->numParams = nparams;

	estate->curr_esql = esql;

	return paramLI;
}

/*
 * returns nth parameter
 *
 */
static void
param_fetch(ParamListInfo params, int paramid)
{
	PLpgPSM_execstate *estate = (PLpgPSM_execstate *) params->paramFetchArg;
	int	dno = paramid - 1;
	ParamExternData		*prm;
	PLpgPSM_datum	*datum;
	PLpgPSM_variable	*var;
	PLpgPSM_variable_ref		*ref;

	ref = (PLpgPSM_variable_ref *) list_nth(estate->curr_esql->var_refs, dno);
	if (ref->ref->type == PLPGPSM_CURSOR)
	{
		PLpgPSM_cursor *cursor = (PLpgPSM_cursor *) ref->ref;
		PLpgPSM_fetch_extra *extra;

		datum = &estate->datums[cursor->offset];
		if (datum->isnull)
			elog(ERROR, "cursor \"%s\" is closed", cursor->name);

		extra = (PLpgPSM_fetch_extra *) datum->extra;

		if (!extra->is_ready)
			elog(ERROR, "cursor \"%s\" is not prepared", cursor->name);

		prm = &params->params[dno];

		prm->value = SPI_getbinval(extra->cursor_tuple, extra->cursor_tupdesc, ref->nattr, &prm->isnull);
		prm->ptype = SPI_gettypeid(extra->cursor_tupdesc, ref->nattr);
	}
	else
	{
		var = (PLpgPSM_variable *) ref->ref;
		Assert(var->astnode.type == PLPGPSM_VARIABLE);

		datum = &estate->datums[var->offset];

		prm = &params->params[dno];

		if (datum->class == PLPGPSM_DATUM_SCALAR)
		{
			prm->value = datum->value;
			prm->isnull = datum->isnull;
			prm->ptype = datum->typeinfo->typeid;
		}
		else if (datum->class == PLPGPSM_DATUM_SQLCODE)
		{
			prm->value = estate->sqlcode;
			prm->isnull = false;
			prm->ptype = datum->typeinfo->typeid;
		}
		else if (datum->class == PLPGPSM_DATUM_SQLSTATE)
		{
			char *buff = unpack_sql_state(estate->sqlcode);
			char *ptr = VARDATA(estate->sqlstate_buffer);

			memcpy(ptr, buff, 5);
			SET_VARSIZE(estate->sqlstate_buffer, 5 + VARHDRSZ);

			prm->value = PointerGetDatum(estate->sqlstate_buffer);
			prm->isnull = false;
			prm->ptype = datum->typeinfo->typeid;
		}
		else
			elog(ERROR, "unsupported class of internal variables");
	}
}

Portal
plpgpsm_open_cursor(PLpgPSM_execstate *estate, PLpgPSM_ESQL *esql, char *name, int cursor_options)
{
	ParamListInfo paramLI;
	SPIPlanPtr	plan;
	Portal		portal;

	if (esql->curr_estate != (psm_estate) estate)
	{
		if (esql->plan != NULL)
		{
			SPI_freeplan(esql->plan);
			esql->plan = NULL;
		}

		if (esql->var_refs != NIL)
		{
			list_free(esql->var_refs);
			esql->var_refs = NIL;
		}
	}

	if (esql->plan == NULL)
	{
		esql->curr_estate = (psm_estate) estate;
		plan = SPI_prepare_params(esql->sqlstr, (ParserSetupHook) plpgpsm_parser_setup, (void *) esql, cursor_options);

		if (plan == NULL)
		{
			/* Some SPI errors deserve specific error messages */
			switch (SPI_result)
			{
				case SPI_ERROR_COPY:
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("cannot COPY to/from client in PL/pgPSM")));
				case SPI_ERROR_TRANSACTION:
					ereport(ERROR,
							(errcode(ERRCODE_FEATURE_NOT_SUPPORTED),
							 errmsg("cannot begin/end transactions in PL/pgPSM"),
							 errhint("Use a BEGIN block with an EXCEPTION clause instead.")));
				default:
					elog(ERROR, "SPI_prepare_params failed for \"%s\": %s",
						 esql->sqlstr, SPI_result_code_string(SPI_result));
			}
		}

		SPI_keepplan(plan);
		esql->plan = plan;
	}

	paramLI = setup_param_list(estate, esql);
	portal = SPI_cursor_open_with_paramlist(name, esql->plan, paramLI, false);

	if (portal == NULL)
		elog(ERROR, "could not open cursor: %s",
				    SPI_result_code_string(SPI_result));

	estate->sqlcode = 0;

	pfree(paramLI);

	return portal;
}

void
plpgpsm_close_cursor(PLpgPSM_execstate *estate, Portal portal)
{
	SPI_cursor_close(portal);
}

void
plpgpsm_cursor_fetch(PLpgPSM_execstate *estate, Portal portal, int prefetch)
{
	SPI_cursor_fetch(portal, true, prefetch);

	estate->eval_processed = SPI_processed;
	estate->eval_tuptable = SPI_tuptable;

	if (SPI_processed > 0)
	{
		estate->sqlcode = 0;
		estate->edata = NULL;
	}
	else
	{
		estate->sqlcode = MAKE_SQLSTATE('0','2','0','0','0');
	}
}

void
plpgpsm_tuple_to_targets(PLpgPSM_execstate *estate, List *datums, HeapTuple tup, TupleDesc tupdesc)
{
	ListCell	*current_datum;
	Datum		*values;
	bool		*nulls;
	int ncolumns;
	int i;

	if (tupdesc == NULL || tup == NULL)
	{
		foreach(current_datum, datums)
		{
			PLpgPSM_datum *datum;
			PLpgPSM_variable *var;
			PLpgPSM_referer *ref;

			ref = (PLpgPSM_referer *) lfirst(current_datum);

			Assert(ref->astnode.type == PLPGPSM_REFERER);
			if (ref->ref->type != PLPGPSM_VARIABLE)
				continue;

			var = (PLpgPSM_variable *) ref->ref;
			datum = &estate->datums[var->offset];

			plpgpsm_set_datum(estate, datum, (Datum) 0, true, false);
		}

		return;
	}

	ncolumns = tupdesc->natts;

	values = (Datum *) palloc(ncolumns * sizeof(Datum));
	nulls = (bool *) palloc(ncolumns * sizeof(bool));

	/* Break down the tuple into fields */
	heap_deform_tuple(tup, tupdesc, values, nulls);

	current_datum = list_head(datums);

	for (i = 0; i < ncolumns; i++)
	{
		Oid	columntyp = tupdesc->attrs[i]->atttypid;
		int32	columntypmod = tupdesc->attrs[i]->atttypmod;
		PLpgPSM_datum *datum;
		PLpgPSM_variable *var;
		PLpgPSM_referer *ref;

		/* Ignore dropped columns */
		if (tupdesc->attrs[i]->attisdropped)
			continue;

		if (current_datum == NULL)
			ereport(ERROR,
					(errcode(ERRCODE_SYNTAX_ERROR),
				errmsg("too few targets")));

		ref = (PLpgPSM_referer *) lfirst(current_datum);
		current_datum = lnext(current_datum);

		Assert(ref->astnode.type == PLPGPSM_REFERER);
		if (ref->ref->type != PLPGPSM_VARIABLE)
			continue;

		var = (PLpgPSM_variable *) ref->ref;
		datum = &estate->datums[var->offset];

		if (!nulls[i])
		{
			PLpgPSM_typeinfo *tinfo = datum->typeinfo;

			if (tinfo->typeid != columntyp || tinfo->typmod != columntypmod)
			{
				char *outstr;
				bool		typIsVarlena;
				Oid		typoutput;
				FmgrInfo		proc;
				Datum		value;

				getTypeOutputInfo(columntyp, &typoutput, &typIsVarlena);
				fmgr_info_cxt(typoutput, &proc, CurrentMemoryContext);
				outstr = OutputFunctionCall(&proc, values[i]);
				value = InputFunctionCall(&tinfo->typinput, outstr, tinfo->typioparam, tinfo->typmod);
				plpgpsm_set_datum(estate, datum, value, false, false);
				pfree(outstr);
			}
			else
			{
				plpgpsm_set_datum(estate, datum, values[i], false, true);
			}
		}
		else
			plpgpsm_set_datum(estate, datum, (Datum) 0, true, false);
	}

	if (current_datum != NULL)
		ereport(ERROR,
				(errcode(ERRCODE_SYNTAX_ERROR),
			errmsg("too many targets")));

	pfree(values);
	pfree(nulls);
}

void
plpgpsm_store_result_to_datums(PLpgPSM_execstate *estate, List *datums)
{
	HeapTuple tup;
	TupleDesc tupdesc;

	if (estate->eval_processed > 0)
	{
		tup = estate->eval_tuptable->vals[0];
		tupdesc = estate->eval_tuptable->tupdesc;

		plpgpsm_tuple_to_targets(estate, datums, tup, tupdesc);
	}
	else
	{
		plpgpsm_tuple_to_targets(estate, datums, NULL, NULL);
	}
}

