/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.store;

import java.io.EOFException;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.LongBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Locale;
import java.util.stream.Collectors;
import org.apache.lucene.store.DataInput;
import org.apache.lucene.store.RandomAccessInput;
import org.apache.lucene.util.Accountable;
import org.apache.lucene.util.GroupVIntUtil;
import org.apache.lucene.util.RamUsageEstimator;

public final class ByteBuffersDataInput
extends DataInput
implements Accountable,
RandomAccessInput {
    private final ByteBuffer[] blocks;
    private final FloatBuffer[] floatBuffers;
    private final LongBuffer[] longBuffers;
    private final int blockBits;
    private final int blockMask;
    private final long length;
    private final long offset;
    private long pos;

    public ByteBuffersDataInput(List<ByteBuffer> buffers) {
        ByteBuffersDataInput.ensureAssumptions(buffers);
        this.blocks = (ByteBuffer[])buffers.toArray(ByteBuffer[]::new);
        for (int i = 0; i < this.blocks.length; ++i) {
            this.blocks[i] = this.blocks[i].asReadOnlyBuffer().order(ByteOrder.LITTLE_ENDIAN);
        }
        this.floatBuffers = new FloatBuffer[this.blocks.length * 4];
        this.longBuffers = new LongBuffer[this.blocks.length * 8];
        if (this.blocks.length == 1) {
            this.blockBits = 32;
            this.blockMask = -1;
        } else {
            int blockBytes = ByteBuffersDataInput.determineBlockPage(buffers);
            this.blockBits = Integer.numberOfTrailingZeros(blockBytes);
            this.blockMask = (1 << this.blockBits) - 1;
        }
        long length = 0L;
        for (ByteBuffer block : this.blocks) {
            length += (long)block.remaining();
        }
        this.length = length;
        this.pos = this.offset = (long)this.blocks[0].position();
    }

    @Override
    public long ramBytesUsed() {
        return (long)RamUsageEstimator.NUM_BYTES_OBJECT_REF * (long)this.blocks.length + Arrays.stream(this.blocks).mapToLong(buf -> buf.capacity()).sum();
    }

    @Override
    public byte readByte() throws EOFException {
        try {
            ByteBuffer block = this.blocks[this.blockIndex(this.pos)];
            byte v = block.get(this.blockOffset(this.pos));
            ++this.pos;
            return v;
        }
        catch (IndexOutOfBoundsException e) {
            if (this.pos >= this.length()) {
                throw new EOFException();
            }
            throw e;
        }
    }

    public void readBytes(ByteBuffer buffer, int len) throws EOFException {
        try {
            while (len > 0) {
                ByteBuffer block = this.blocks[this.blockIndex(this.pos)].duplicate();
                int blockOffset = this.blockOffset(this.pos);
                block.position(blockOffset);
                int chunk = Math.min(len, block.remaining());
                if (chunk == 0) {
                    throw new EOFException();
                }
                this.pos += (long)chunk;
                block.limit(blockOffset + chunk);
                buffer.put(block);
                len -= chunk;
            }
        }
        catch (ArrayIndexOutOfBoundsException | BufferUnderflowException e) {
            if (this.pos >= this.length()) {
                throw new EOFException();
            }
            throw e;
        }
    }

    @Override
    public void readBytes(byte[] arr, int off, int len) throws EOFException {
        try {
            while (len > 0) {
                ByteBuffer block = this.blocks[this.blockIndex(this.pos)].duplicate();
                block.position(this.blockOffset(this.pos));
                int chunk = Math.min(len, block.remaining());
                if (chunk == 0) {
                    throw new EOFException();
                }
                this.pos += (long)chunk;
                block.get(arr, off, chunk);
                len -= chunk;
                off += chunk;
            }
        }
        catch (ArrayIndexOutOfBoundsException | BufferUnderflowException e) {
            if (this.pos >= this.length()) {
                throw new EOFException();
            }
            throw e;
        }
    }

    @Override
    public short readShort() throws IOException {
        int blockOffset = this.blockOffset(this.pos);
        if (blockOffset + 2 <= this.blockMask) {
            short v = this.blocks[this.blockIndex(this.pos)].getShort(blockOffset);
            this.pos += 2L;
            return v;
        }
        return super.readShort();
    }

    @Override
    public int readInt() throws IOException {
        int blockOffset = this.blockOffset(this.pos);
        if (blockOffset + 4 <= this.blockMask) {
            int v = this.blocks[this.blockIndex(this.pos)].getInt(blockOffset);
            this.pos += 4L;
            return v;
        }
        return super.readInt();
    }

    @Override
    public long readLong() throws IOException {
        int blockOffset = this.blockOffset(this.pos);
        if (blockOffset + 8 <= this.blockMask) {
            long v = this.blocks[this.blockIndex(this.pos)].getLong(blockOffset);
            this.pos += 8L;
            return v;
        }
        return super.readLong();
    }

    @Override
    public void readGroupVInt(int[] dst, int offset) throws IOException {
        ByteBuffer block = this.blocks[this.blockIndex(this.pos)];
        int blockOffset = this.blockOffset(this.pos);
        int len = GroupVIntUtil.readGroupVInt((DataInput)this, (long)(block.limit() - blockOffset), p -> block.getInt((int)p), (long)blockOffset, dst, offset);
        this.pos += (long)len;
    }

    @Override
    public long length() {
        return this.length;
    }

    @Override
    public byte readByte(long pos) {
        return this.blocks[this.blockIndex(pos += this.offset)].get(this.blockOffset(pos));
    }

    @Override
    public void readBytes(long pos, byte[] bytes, int offset, int len) throws IOException {
        long absPos = this.offset + pos;
        try {
            while (len > 0) {
                ByteBuffer block = this.blocks[this.blockIndex(absPos)];
                int blockPosition = this.blockOffset(absPos);
                int chunk = Math.min(len, block.capacity() - blockPosition);
                if (chunk == 0) {
                    throw new EOFException();
                }
                block.get(blockPosition, bytes, offset, chunk);
                absPos += (long)chunk;
                len -= chunk;
                offset += chunk;
            }
        }
        catch (ArrayIndexOutOfBoundsException | BufferUnderflowException e) {
            if (absPos >= this.length()) {
                throw new EOFException();
            }
            throw e;
        }
    }

    @Override
    public short readShort(long pos) {
        long absPos = this.offset + pos;
        int blockOffset = this.blockOffset(absPos);
        if (blockOffset + 2 <= this.blockMask) {
            return this.blocks[this.blockIndex(absPos)].getShort(blockOffset);
        }
        return (short)(this.readByte(pos) & 0xFF | (this.readByte(pos + 1L) & 0xFF) << 8);
    }

    @Override
    public int readInt(long pos) {
        long absPos = this.offset + pos;
        int blockOffset = this.blockOffset(absPos);
        if (blockOffset + 4 <= this.blockMask) {
            return this.blocks[this.blockIndex(absPos)].getInt(blockOffset);
        }
        return this.readByte(pos) & 0xFF | (this.readByte(pos + 1L) & 0xFF) << 8 | (this.readByte(pos + 2L) & 0xFF) << 16 | this.readByte(pos + 3L) << 24;
    }

    @Override
    public long readLong(long pos) {
        long absPos = this.offset + pos;
        int blockOffset = this.blockOffset(absPos);
        if (blockOffset + 8 <= this.blockMask) {
            return this.blocks[this.blockIndex(absPos)].getLong(blockOffset);
        }
        byte b1 = this.readByte(pos);
        byte b2 = this.readByte(pos + 1L);
        byte b3 = this.readByte(pos + 2L);
        byte b4 = this.readByte(pos + 3L);
        byte b5 = this.readByte(pos + 4L);
        byte b6 = this.readByte(pos + 5L);
        byte b7 = this.readByte(pos + 6L);
        byte b8 = this.readByte(pos + 7L);
        return ((long)b8 & 0xFFL) << 56 | ((long)b7 & 0xFFL) << 48 | ((long)b6 & 0xFFL) << 40 | ((long)b5 & 0xFFL) << 32 | ((long)b4 & 0xFFL) << 24 | ((long)b3 & 0xFFL) << 16 | ((long)b2 & 0xFFL) << 8 | (long)b1 & 0xFFL;
    }

    @Override
    public void readFloats(float[] arr, int off, int len) throws EOFException {
        try {
            while (len > 0) {
                FloatBuffer floatBuffer = this.getFloatBuffer(this.pos);
                floatBuffer.position(this.blockOffset(this.pos) >> 2);
                int chunk = Math.min(len, floatBuffer.remaining());
                if (chunk == 0) {
                    arr[off] = Float.intBitsToFloat(this.readInt(this.pos - this.offset));
                    ++off;
                    --len;
                    this.pos += 4L;
                    continue;
                }
                this.pos += (long)(chunk << 2);
                floatBuffer.get(arr, off, chunk);
                len -= chunk;
                off += chunk;
            }
        }
        catch (IndexOutOfBoundsException | BufferUnderflowException e) {
            if (this.pos - this.offset + 4L > this.length()) {
                throw new EOFException();
            }
            throw e;
        }
    }

    @Override
    public void readLongs(long[] arr, int off, int len) throws EOFException {
        try {
            while (len > 0) {
                LongBuffer longBuffer = this.getLongBuffer(this.pos);
                longBuffer.position(this.blockOffset(this.pos) >> 3);
                int chunk = Math.min(len, longBuffer.remaining());
                if (chunk == 0) {
                    arr[off] = this.readLong(this.pos - this.offset);
                    ++off;
                    --len;
                    this.pos += 8L;
                    continue;
                }
                this.pos += (long)(chunk << 3);
                longBuffer.get(arr, off, chunk);
                len -= chunk;
                off += chunk;
            }
        }
        catch (IndexOutOfBoundsException | BufferUnderflowException e) {
            if (this.pos - this.offset + 8L > this.length()) {
                throw new EOFException();
            }
            throw e;
        }
    }

    private FloatBuffer getFloatBuffer(long pos) {
        int alignment;
        int bufferIndex = this.blockIndex(pos);
        int floatBufferIndex = bufferIndex * 4 + (alignment = (int)pos & 3);
        if (this.floatBuffers[floatBufferIndex] == null) {
            ByteBuffer dup = this.blocks[bufferIndex].duplicate();
            dup.position(alignment);
            this.floatBuffers[floatBufferIndex] = dup.order(ByteOrder.LITTLE_ENDIAN).asFloatBuffer();
        }
        return this.floatBuffers[floatBufferIndex];
    }

    private LongBuffer getLongBuffer(long pos) {
        int alignment;
        int bufferIndex = this.blockIndex(pos);
        int longBufferIndex = bufferIndex * 8 + (alignment = (int)pos & 7);
        if (this.longBuffers[longBufferIndex] == null) {
            ByteBuffer dup = this.blocks[bufferIndex].duplicate();
            dup.position(alignment);
            this.longBuffers[longBufferIndex] = dup.order(ByteOrder.LITTLE_ENDIAN).asLongBuffer();
        }
        return this.longBuffers[longBufferIndex];
    }

    public long position() {
        return this.pos - this.offset;
    }

    public void seek(long position) throws EOFException {
        this.pos = position + this.offset;
        if (position > this.length()) {
            this.pos = this.length();
            throw new EOFException();
        }
    }

    @Override
    public void skipBytes(long numBytes) throws IOException {
        if (numBytes < 0L) {
            throw new IllegalArgumentException("numBytes must be >= 0, got " + numBytes);
        }
        long skipTo = this.position() + numBytes;
        this.seek(skipTo);
    }

    public ByteBuffersDataInput slice(long offset, long length) {
        if (offset < 0L || length < 0L || offset + length > this.length) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "slice(offset=%s, length=%s) is out of bounds: %s", offset, length, this));
        }
        return new ByteBuffersDataInput(ByteBuffersDataInput.sliceBufferList(Arrays.asList(this.blocks), offset, length));
    }

    public String toString() {
        return String.format(Locale.ROOT, "%,d bytes, block size: %,d, blocks: %,d, position: %,d%s", this.length(), this.blockSize(), this.blocks.length, this.position(), this.offset == 0L ? "" : String.format(Locale.ROOT, " [offset: %,d]", this.offset));
    }

    private final int blockIndex(long pos) {
        return Math.toIntExact(pos >> this.blockBits);
    }

    private final int blockOffset(long pos) {
        return (int)pos & this.blockMask;
    }

    private int blockSize() {
        return 1 << this.blockBits;
    }

    private static final boolean isPowerOfTwo(int v) {
        return (v & v - 1) == 0;
    }

    private static void ensureAssumptions(List<ByteBuffer> buffers) {
        if (buffers.isEmpty()) {
            throw new IllegalArgumentException("Buffer list must not be empty.");
        }
        if (buffers.size() != 1) {
            int blockPage = ByteBuffersDataInput.determineBlockPage(buffers);
            if (!ByteBuffersDataInput.isPowerOfTwo(blockPage)) {
                throw new IllegalArgumentException("The first buffer must have power-of-two position() + remaining(): 0x" + Integer.toHexString(blockPage));
            }
            int last = buffers.size() - 1;
            for (int i = 1; i < last; ++i) {
                ByteBuffer buffer = buffers.get(i);
                if (buffer.position() != 0) {
                    throw new IllegalArgumentException("All buffers except for the first one must have position() == 0: " + String.valueOf(buffer));
                }
                if (i == last || buffer.remaining() == blockPage) continue;
                throw new IllegalArgumentException("Intermediate buffers must share an identical remaining() power-of-two block size: 0x" + Integer.toHexString(blockPage));
            }
        }
    }

    static int determineBlockPage(List<ByteBuffer> buffers) {
        ByteBuffer first = buffers.get(0);
        int blockPage = Math.toIntExact((long)first.position() + (long)first.remaining());
        return blockPage;
    }

    private static List<ByteBuffer> sliceBufferList(List<ByteBuffer> buffers, long offset, long length) {
        ByteBuffersDataInput.ensureAssumptions(buffers);
        if (buffers.size() == 1) {
            ByteBuffer cloned = buffers.get(0).asReadOnlyBuffer().order(ByteOrder.LITTLE_ENDIAN);
            cloned.position(Math.toIntExact((long)cloned.position() + offset));
            cloned.limit(Math.toIntExact((long)cloned.position() + length));
            return Arrays.asList(cloned);
        }
        long absStart = (long)buffers.get(0).position() + offset;
        long absEnd = absStart + length;
        int blockBytes = ByteBuffersDataInput.determineBlockPage(buffers);
        int blockBits = Integer.numberOfTrailingZeros(blockBytes);
        long blockMask = (1L << blockBits) - 1L;
        int endOffset = Math.toIntExact(absEnd & blockMask);
        ArrayList cloned = buffers.subList(Math.toIntExact(absStart / (long)blockBytes), Math.toIntExact(absEnd / (long)blockBytes + (long)(endOffset == 0 ? 0 : 1))).stream().map(buf -> buf.asReadOnlyBuffer().order(ByteOrder.LITTLE_ENDIAN)).collect(Collectors.toCollection(ArrayList::new));
        if (endOffset == 0) {
            cloned.add(ByteBuffer.allocate(0).order(ByteOrder.LITTLE_ENDIAN));
        }
        ((ByteBuffer)cloned.get(0)).position(Math.toIntExact(absStart & blockMask));
        ((ByteBuffer)cloned.get(cloned.size() - 1)).limit(endOffset);
        return cloned;
    }
}

