
/******************************************************
 *
 * KCT-T project.
 *
 * Jean-Paul ROUBELAT - F6FBB - jpr@f6fbb.org
 *
 * Version 0.1 14/03/99 : Initial release
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************/

/******************************************************
 *                                                    *
 * kctlib.h                                           *
 *                                                    *
 * kct library headers and prototypes                 *
 *                                                    *
 ******************************************************/

#include <linux/kctt.h>

#define KCTT_CONF "/etc/ax25/rotor.conf"

#define KCTT_UNKNOWN 0
#define KCTT_ICOM 1
#define KCTT_YAESU 2
#define KCTT_KENWOOD 3

typedef int kctdev;

/******************************************************
 * Structure kctstat                                  *
 *                                                    *
 * int site_pos:                                      *
 *   Position of the site rotor (0..255)              *
 *                                                    *
 * int site_win:                                      *
 *   Width of the site window (0..25)                 *
 *                                                    *
 * int site_brkset:                                   *
 *   Delay of site brake set (0..25500 ms)            *
 *                                                    *
 * int site_brkrel:                                   *
 *   Delay of site brake release (0..25500 ms)        *
 *                                                    *
 * int site_pos:                                      *
 *   Position of the azimuth rotor (0..255)           *
 *                                                    *
 * int site_win;                                      *
 *   Width of the azimuth window (0..25)              *
 *                                                    *
 * int site_brkset:                                   *
 *   Delay of azimuth brake set (0..25500 ms)         *
 *                                                    *
 * int site_brkrel:                                   *
 *   Delay of azimuth brake release (0..25500 ms)     *
 *                                                    *
 * int control:                                       *
 *   b7 = Azimuth brake released                      *
 *   b5 = Azimuth move requested                      *
 *   b4 = Site move requested                         *
 *   b3 = Rotor left                                  *
 *   b2 = Rotor right                                 *
 *   b1 = Rotor down                                  *
 *   b0 = Rotor up                                    *
 ******************************************************/

typedef struct
{
	int site_pos;		/* 0 .. 255	*/
	int site_win;		/* 0 .. 25 */
	int site_brkset;	/* 0 .. 25500 ms */
	int site_brkrel;	/* 0 .. 25500 ms */
	int azim_pos;		/* 0 .. 255 */
	int azim_win;		/* 0 .. 25 */
	int azim_brkset;	/* 0 .. 25500 ms */
	int azim_brkrel;	/* 0 .. 25500 ms */
	int control;
} kctstat;

/******************************************************
 *                                                    *
 * prototypes                                         *
 *                                                    *
 ******************************************************/
 
/* tracker.c */

/******************************************************
 * KctOpen                                            *
 *   Description :                                    *
 *     Open a kansas city device                      *
 *   Arguments :                                      *
 *     int bord : number of the board (first = 0)     *
 *   Return value :                                   *
 *     the open kct device                            *
 ******************************************************/
kctdev KctOpen(int board);

/******************************************************
 * KctClose                                           *
 *   Description :                                    *
 *     Close a kansas city device                     *
 *   Arguments :                                      *
 *     kctdev dev : the open kct device               *
 *   Return value :                                   *
 *     none                                           *
 ******************************************************/
void KctClose(kctdev dev);

/******************************************************
 * KctSetBrake                                        *
 *   Description :                                    *
 *     Set the brake parameters                       *
 *   Arguments :                                      *
 *     kctdev dev : the open kct device               *
 *     int rotor : KCT_AZIMUTH or KCT_SITE            *
 *     int tempo_rel :                                *
 *                release temporisation (0..25500 ms) *
 *     int tempo_set :                                *
 *                set temporisation (0..25500 ms)     *
 *   Return value :                                   *
 *     > 0 if OK                                      *
 ******************************************************/
int KctSetBrake(kctdev dev, int rotor, int tempo_rel, int tempo_set);

/******************************************************
 * KctSetWindow                                       *
 *   Description :                                    *
 *     Set the window size. Rotor will be started     *
 *     if the new value is out of the window size     *
 *   Arguments :                                      *
 *     kctdev dev : the open kct device               *
 *     int port : KCT_AZIMUTH or KCT_SITE             *
 *     int val : window size (0..25)                  *
 *   Return value :                                   *
 *     > 0 if OK                                      *
 ******************************************************/
int KctSetWindow(kctdev dev, int port, int val);

/******************************************************
 * KctSetPosition                                     *
 *   Description :                                    *
 *     Set the rotor position                         *
 *   Arguments :                                      *
 *     kctdev dev : the open kct device               *
 *     int rotor : KCT_AZIMUTH or KCT_SITE            *
 *     int pos : requiered rotor position (0..255)    *
 *   Return value :                                   *
 *     > 0 if OK                                      *
 ******************************************************/
int KctSetPosition(kctdev dev, int rotor, int pos);

/******************************************************
 * KctGetStatus                                       *
 *   Description :                                    *
 *     Get the device status                          *
 *   Arguments :                                      *
 *     kctdev dev : the open kct device               *
 *     KctGetStatus : struct filled with information  *
 *   Return value :                                   *
 *     > 0 if OK                                      *
 ******************************************************/
int KctGetStatus(kctdev dev, kctstat *stat);

int KctSetSite(kctdev dev, double site);
int KctSetAzimuth(kctdev dev, double azim);
int KctSetSiteOrigin(kctdev dev, int orig, int min, int max);
int KctSetAzimuthOrigin(kctdev dev, int orig, int min, int max);
double KctGetSite(kctdev dev);
double KctGetAzimuth(kctdev dev);

int KctSetFrequency(kctdev dev, int rig, double freq);

/* icom.c */
int KctSetFrequencyIcom(char *buf, int buflen, int addr, double freq);
