/*
    Usage: sendkiss -f infile pty

    Read a file containing KISS records and sends each frame
    to the named pseudo tty. When sendkiss has been started,
    the pseudo tty may be connected to a socket with axattach

    Output received from the port is simply discarded

 */

char RCD_ID [] = "$Id: sendkiss.c,v 1.3 2000/06/06 17:53:59 oz6bl Exp $" ;
char version [] = "Sendkiss V0.9 - (c) Bent Bagger, OZ6BL" ;
char usage_string [] = "usage: sendkiss -f infile pty\n" ;

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <termios.h>
#include <fcntl.h>
#include <signal.h>
#include <ctype.h>
#include <getopt.h>
#include <errno.h>

#include <netax25/ttyutils.h>

#define FR_END   0xc0
#define FR_ESC   0xdb
#define T_FR_END 0xdc
#define T_FR_ESC 0xdd
#define FEND FR_END
#define FESC FR_ESC
#define TFEND T_FR_END
#define TFESC T_FR_ESC

#define SIZE 4096

static unsigned char obuf [SIZE] ;
static unsigned char ibuf [SIZE] ;
struct iface
{
   char            *name;          /* Interface name (/dev/???)    */
   int             fd;             /* File descriptor              */
   int             escaped;        /* FESC received?               */
   unsigned char   crc;            /* Incoming frame crc           */
   unsigned char   obuf[SIZE];     /* TX buffer                    */
   unsigned char   *optr;          /* Next byte to transmit        */
   unsigned int    errors;         /* KISS protocol error count    */
   unsigned int    rxpackets;      /* RX frames count              */
   unsigned int    txpackets;      /* TX frames count              */
   unsigned long   rxbytes;        /* RX bytes count               */
   unsigned long   txbytes;        /* TX bytes count               */
};

static int kiss_tx(int fd, int port, unsigned char *s, int len, int crcflag)
{
        unsigned char *ptr = obuf;
        unsigned char c;
        unsigned char crc = 0;
        int first = TRUE;

        /* Allow for checksum byte */
        if (crcflag)
                len++;

        /*
         * Send an initial FEND character to flush out any
         * data that may have accumulated in the receiver
         * due to line noise.
         */

        *ptr++ = FEND;

        /*
         * For each byte in the packet, send the appropriate
         * character sequence, according to the SLIP protocol.
         */

        while (len-- > 0) {
                c = *s++;
                if (first == TRUE) { /* Control byte */
                        c = (c & 0x0F) | (port << 4);
                        first = FALSE;
                }
                if (crcflag) {
                        if (len == 0)           /* Now past user data...   */
                                c = crc;        /* ...time to encode cksum */
                        else
                                crc ^= c;       /* Adjust checksum */
                }
                switch (c) {
                      case FEND:
                        *ptr++ = FESC;
                        *ptr++ = TFEND;
                        break;
                      case FESC:
                        *ptr++ = FESC;
                        *ptr++ = TFESC;
                        break;
                      default:
                        *ptr++ = c;
                        break;
                }
        }
        *ptr++ = FEND;
        return write(fd, obuf, ptr - obuf);
}

/* Read a kiss packet into buffer */
int get_kiss_frame(FILE *fp, char buffer[], int nbyte)
{
	int c, count = 0 ;

	while ((c = getc (fp)) != EOF && c != FR_END) ; /* find beginning of frame */
	while (nbyte-- > 0 && (c = getc(fp)) != EOF) {
		switch (c) {
		case FR_ESC:
			switch (c = getc(fp)) {
			case T_FR_ESC:
				buffer[count++] = FR_ESC ;
				break ;
			case T_FR_END:
				buffer[count++] = FR_END ;
				break ; }
			break ;
		case FR_END:
			return (count) ;
		default:
			buffer[count++] = c ;
			break ; }
	}
	while ((c = getc (fp)) != EOF && c != FR_END) ; /* find end of frame */
	return (EOF) ;
}

int main(int argc, char **argv)
{
	struct iface *pty ;
	int i, size ;
	FILE * input_file ;
	char *input_file_name = NULL ;
#define MAXBUFFER 1000  
	char buffer [MAXBUFFER] ;
	fd_set readfs ;

	while ((i = getopt(argc, argv, "vf:")) != -1) {
		switch (i) {
			case 'v':
				printf("sendkiss %s\n", version);
				return 0;
			case 'f':
				input_file_name = optarg ;
				break ;
			case '?':
			case ':':
				fprintf(stderr, usage_string) ;
				return 1;
		}
	}
	if (argc < 2) {
		fprintf(stderr, usage_string);
		return 1; }
	/*
	 * Check for lock files before opening any TTYs
	 */
	if (tty_is_locked(argv[optind])) {
		fprintf(stderr, "sendkiss: pty %s is locked by another process\n", argv[optind]) ; 
		return 1; }
	/*
	 * Open and configure the pty interfaces
	 */
	if ((pty = calloc (1, sizeof(struct iface))) == NULL) {
		perror("sendkiss: malloc");
		return 1; }
	if ((pty->fd = open(argv[optind], O_RDWR)) == -1) {
		perror("sendkiss: open");
		return 1; }
	pty->name = argv[optind];
	tty_raw(pty->fd, FALSE);
	pty->optr = pty->obuf;
	/*
	 * Now all the ports are open, lock them.
	 */
	tty_lock(argv[optind]);

	if (input_file_name == NULL) input_file = stdin ;
	else {
		if ((input_file = fopen (input_file_name, "rb")) == NULL) {
			perror ("sendkiss: open input") ;
			return 1 ; } }
	while (TRUE) {
		FD_ZERO (&readfs) ;
		FD_SET (pty->fd, &readfs) ;
		FD_SET (fileno (input_file), &readfs) ;
		select (32, &readfs, NULL, NULL, NULL) ;
		/*
		 * Something has happened
		 */
		if (FD_ISSET ( fileno (input_file), &readfs)) {
			if ((i = get_kiss_frame(input_file, buffer, MAXBUFFER)) > 0) {
				if (kiss_tx (pty->fd, 0, buffer, i, FALSE) < 0) {
					perror ("sendkiss: kiss_tx") ;
					return 1 ; }
				pty->txpackets++ ;
				pty->txbytes += i ; }
 			else /* EOF */ {
				fclose (input_file) ;
				if ((input_file = fopen (input_file_name, "rb")) == NULL) {
					perror ("sendkiss: open input") ;
					return 1 ; } } }
		if (FD_ISSET (pty->fd, &readfs)) {
			if ((size = read (pty->fd, ibuf, SIZE)) < 0 && errno != EINTR) {
				perror ("sendkiss: read pty") ; 
				return 1 ;  }
			else {
				/* printf ("size %d\n", size) */ ; }
		}
	}
}
