/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.camel.quarkus.component.influxdb.it;

import java.util.Map;

import io.quarkus.test.common.QuarkusTestResourceLifecycleManager;
import org.apache.camel.util.CollectionHelper;
import org.eclipse.microprofile.config.ConfigProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.testcontainers.containers.GenericContainer;
import org.testcontainers.containers.wait.strategy.Wait;

public class InfluxdbTestResource implements QuarkusTestResourceLifecycleManager {
    public static final Logger LOGGER = LoggerFactory.getLogger(InfluxdbTestResource.class);
    public static final int INFLUXDB_PORT = 8086;
    public static final String INFLUXDB_IMAGE = ConfigProvider.getConfig().getValue("influxdb.container.image", String.class);
    public static final String INFLUXDB_VERSION = INFLUXDB_IMAGE.split(":")[1];

    private GenericContainer<?> container;

    @Override
    public Map<String, String> start() {
        try {
            container = new GenericContainer<>(INFLUXDB_IMAGE)
                    .withExposedPorts(INFLUXDB_PORT)
                    .waitingFor(Wait.forListeningPort());

            container.start();

            return CollectionHelper.mapOf(
                    InfluxdbResource.INFLUXDB_CONNECTION_PROPERTY,
                    "http://" + String.format("%s:%s", container.getHost(),
                            container.getMappedPort(INFLUXDB_PORT)));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public void stop() {
        try {
            if (container != null) {
                container.stop();
            }
        } catch (Exception e) {
            // ignored
        }
    }
}
