/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.operations.command;

import org.apache.flink.annotation.Internal;
import org.apache.flink.table.api.TableException;
import org.apache.flink.table.api.internal.TableResultImpl;
import org.apache.flink.table.api.internal.TableResultInternal;
import org.apache.flink.table.operations.ExecutableOperation;
import org.apache.flink.table.operations.Operation;
import org.apache.flink.table.resource.ResourceType;
import org.apache.flink.table.resource.ResourceUri;

import java.io.IOException;
import java.util.Collections;

/** Operation to describe an ADD JAR statement. */
@Internal
public class AddJarOperation implements Operation, ExecutableOperation {

    private final String path;

    public AddJarOperation(String path) {
        this.path = path;
    }

    public String getPath() {
        return path;
    }

    @Override
    public String asSummaryString() {
        return String.format("ADD JAR '%s'", path);
    }

    @Override
    public TableResultInternal execute(Context ctx) {
        ResourceUri resourceUri = new ResourceUri(ResourceType.JAR, getPath());
        try {
            ctx.getResourceManager().registerJarResources(Collections.singletonList(resourceUri));
            return TableResultImpl.TABLE_RESULT_OK;
        } catch (IOException e) {
            throw new TableException(
                    String.format("Could not register the specified resource [%s].", resourceUri),
                    e);
        }
    }
}
