/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <pwd.h>
#include <time.h>

#include <sys/types.h>
#include <sys/uio.h>
#include <sys/socket.h>

#include <arpa/inet.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#include "log.h"
#include "handler.h"
#include "connection.h"
#include "buffer.h"
#include "handler_table_entry.h"
#include "encoder_table.h"
#include "server.h"


ret_t
cherokee_connection_new  (cherokee_connection_t **cnt)
{
	CHEROKEE_NEW_STRUCT(n, connection);
	   
	INIT_LIST_HEAD(&n->list_entry);

	n->tcp_cork       = 0;
	n->error_code     = http_ok;
	n->phase          = phase_reading_header;
	n->handler        = NULL; 
	n->encoder        = NULL;
	n->encoder_buffer = NULL;
	n->arguments      = NULL;
	n->logger_ref     = NULL;
	n->keepalive      = 0;
	n->range_start    = 0;
	n->range_end      = 0;
	n->vserver        = NULL;
	n->log_at_end     = 0;
	n->realm_ref      = NULL;
	n->mmaped         = NULL;
	n->thread         = NULL;

	n->post        = NULL;
	n->post_len    = 0;

	cherokee_buffer_new (&n->buffer);
	cherokee_buffer_new (&n->header_buffer);
	cherokee_buffer_new (&n->incoming_header);

	cherokee_buffer_new (&n->local_directory);
	cherokee_buffer_new (&n->web_directory);
	cherokee_buffer_new (&n->userdir);
	cherokee_buffer_new (&n->request);
	cherokee_buffer_new (&n->redirect);
	cherokee_buffer_new (&n->host);
	cherokee_buffer_new (&n->query_string);
	cherokee_buffer_new (&n->user);
	cherokee_buffer_new (&n->passwd);
	
	cherokee_socket_new (&n->socket);
	cherokee_header_new (&n->header);

	*cnt = n;
	return ret_ok;
}


ret_t
cherokee_connection_free (cherokee_connection_t  *cnt)
{
	cherokee_header_free (cnt->header);
	cherokee_socket_free (cnt->socket);
	
	if (cnt->handler != NULL) {
		cherokee_handler_free (cnt->handler);
		cnt->handler = NULL;
	}

	if (cnt->encoder != NULL) {
		cherokee_encoder_free (cnt->encoder);
		cnt->encoder = NULL;
	}

	if (cnt->encoder_buffer != NULL) {
		cherokee_buffer_free (cnt->encoder_buffer);
		cnt->encoder_buffer = NULL;
	}

	if (cnt->post != NULL) {
		cherokee_buffer_free (cnt->post);
		cnt->post = NULL;
	}
	
	cherokee_buffer_free (cnt->buffer);
	cherokee_buffer_free (cnt->header_buffer);
	cherokee_buffer_free (cnt->incoming_header);

	cherokee_buffer_free (cnt->local_directory);
	cherokee_buffer_free (cnt->web_directory);
	cherokee_buffer_free (cnt->userdir);
	cherokee_buffer_free (cnt->request);
	cherokee_buffer_free (cnt->redirect);
	cherokee_buffer_free (cnt->host);
	cherokee_buffer_free (cnt->user);
	cherokee_buffer_free (cnt->passwd);
	
	if (cnt->arguments) {
		cherokee_table_free2 (cnt->arguments, free);
		cnt->arguments = NULL;
	}
	
	free (cnt);
	return ret_ok;
}


ret_t
cherokee_connection_clean (cherokee_connection_t *cnt)
{	   
	uint32_t header_len;

	cnt->phase        = phase_reading_header;
	cnt->error_code   = http_ok;
	cnt->timeout      = -1;
	cnt->range_start  = 0;
	cnt->range_end    = 0;
	cnt->logger_ref   = NULL;
	cnt->tcp_cork     = 0;
	cnt->log_at_end   = 0;
	cnt->realm_ref    = NULL;
	cnt->post_len     = 0;
	cnt->mmaped       = NULL;
	
	if (cnt->handler != NULL) {
		cherokee_handler_free (cnt->handler);
		cnt->handler = NULL;
	}
	
	if (cnt->encoder != NULL) {
		cherokee_encoder_free (cnt->encoder);
		cnt->encoder = NULL;
	}

	if (cnt->encoder_buffer != NULL) {
		cherokee_buffer_free (cnt->encoder_buffer);
		cnt->encoder_buffer = NULL;
	}

	if (cnt->post != NULL) {
		cherokee_buffer_free (cnt->post);
		cnt->post = NULL;
	}
	
	cherokee_buffer_clean (cnt->local_directory);
	cherokee_buffer_clean (cnt->web_directory);
	cherokee_buffer_clean (cnt->userdir);
	cherokee_buffer_clean (cnt->request);
	cherokee_buffer_clean (cnt->redirect);
	cherokee_buffer_clean (cnt->host);
	cherokee_buffer_clean (cnt->query_string);
	cherokee_buffer_clean (cnt->user);
	cherokee_buffer_clean (cnt->passwd);
	
	if (cnt->arguments) {
		cherokee_table_free2 (cnt->arguments, free);
		cnt->arguments = NULL;
	}

	/* Drop out the last incoming header
	 */
	cherokee_header_get_length (cnt->header, &header_len);
	
	cherokee_header_clean (cnt->header);
	cherokee_buffer_clean (cnt->buffer);
	cherokee_buffer_clean (cnt->header_buffer);
	cherokee_buffer_move_to_begin (cnt->incoming_header, header_len);

	/* If the connection has incoming headers to be processed,
	 * then increment the pending counter from the thread
	 */	 
	if (! cherokee_buffer_is_empty (cnt->incoming_header)) {
		CONN_THREAD(cnt)->pending_conns_num++;
	}

	cnt->thread = NULL;

	return ret_ok;
}


ret_t 
cherokee_connection_mrproper (cherokee_connection_t *cnt)
{
	ret_t ret;

	cnt->keepalive = 0;

	ret  = cherokee_socket_close (cnt->socket);
	ret |= cherokee_connection_clean (cnt);

	return ret;
}


static inline void
add_error_code_string_to_buffer (cherokee_connection_t *cnt)
{
	switch(cnt->error_code) {
	case http_ok:
		cherokee_buffer_add (cnt->buffer, http_ok_string, 6); break;
	case http_accepted:
		cherokee_buffer_add (cnt->buffer, http_accepted_string CRLF, 12); break;
	case http_partial_content:
		cherokee_buffer_add (cnt->buffer, http_partial_content_string CRLF, 19); break;		
	case http_bad_request:
		cherokee_buffer_add (cnt->buffer, http_bad_request_string CRLF, 15); break;
	case http_access_denied:
		cherokee_buffer_add (cnt->buffer, http_access_denied_string CRLF, 13); break;
	case http_not_found:
		cherokee_buffer_add (cnt->buffer, http_not_found_string CRLF, 13); break;
	case http_internal_error:
		cherokee_buffer_add (cnt->buffer, http_internal_error_string CRLF, 25); break;
	case http_moved_permanently: 
		cherokee_buffer_add (cnt->buffer, http_moved_permanently_string CRLF, 21); break;
	case http_moved_temporarily:
		cherokee_buffer_add (cnt->buffer, http_moved_temporarily_string CRLF, 21); break;	
	case http_unauthorized:
		cherokee_buffer_add (cnt->buffer, http_unauthorized_string CRLF, 26); break;
	case http_not_modified:
		cherokee_buffer_add (cnt->buffer, http_not_modified_string CRLF, 16); break;	
	default:
		SHOULDNT_HAPPEN;
	}
}


static ret_t
send_buffer_unsafe (cherokee_connection_t *cnt)
{
	ret_t ret;
	int   sent;

//	ret = write (SOCKET_FD(cnt->socket), cnt->buffer->buf, cnt->buffer->len);
	ret = cherokee_socket_write (cnt->socket, cnt->buffer, &sent);

	if (ret < ret_ok) {
		cnt->keepalive = 0;
		return ret_error;
	}

	/* Add to the virtual host data sent counter
	 */
	CONN_VSRV(cnt)->data.tx += ret;

	return ret_ok;
}


static ret_t 
process_handler_complex_headers (cherokee_connection_t *cnt)
{
	ret_t ret = ret_ok;
	char *info;
	int   info_len;
	cherokee_header_t *header;
	
	cherokee_header_new (&header);
	
	ret = cherokee_header_parse (header, cnt->header_buffer, header_type_basic);
	if (ret < ret_ok) {
		return ret;
	}
	
	/* Location: 
	 */
	if (cherokee_header_get_known (header, header_location, &info, &info_len) == ret_ok) {
		cherokee_buffer_make_empty (cnt->redirect);
		cherokee_header_copy_known (header, header_location, cnt->redirect);
		cnt->error_code = http_moved_permanently;
	}
	
	/* Status:
	 */
	if (cherokee_header_get_unknown (header, "Status", 6, &info, &info_len) == ret_ok) {
		int r;
		int code;
		CHEROKEE_TEMP(status,4);
		
		memcpy (status, info, 3);
		status[3] = '\0';
		
		r = sscanf (status, "%d", &code);
		if (r <= 0) {
			ret = ret_error;
			goto exit;
		}

		cnt->error_code = code;
		return ret_error;
	}
	
exit:
	cherokee_header_free (header);
	return ret;
}


static void
build_response_header (cherokee_connection_t *cnt)
{	
	cherokee_buffer_make_empty (cnt->buffer);
	
	/* Add protocol string + error_code
	 */
	switch (cnt->header->version) {
	case http_version_09:
		cherokee_buffer_add (cnt->buffer, "HTTP/0.9 ", 9); 
		break;
	case http_version_11:
		cherokee_buffer_add (cnt->buffer, "HTTP/1.1 ", 9); 
		break;
	case http_version_10:
	default:
		cherokee_buffer_add (cnt->buffer, "HTTP/1.0 ", 9); 
		break;
	}
	
	add_error_code_string_to_buffer (cnt);
	cherokee_buffer_add (cnt->buffer, CRLF, 2); 

	/* Add server name
	 */
	if (CONN_SRV(cnt)->hideservername == 0) {
		if (CONN_SRV(cnt)->hideversion) {
			cherokee_buffer_add (cnt->buffer, "Server: Cherokee" CRLF, 16+2);
		} else {
//			cherokee_buffer_add (cnt->buffer, "Server: Cherokee/" VERSION CRLF, 17+5+2);
// ToFix: >=0.4.10
			cherokee_buffer_add (cnt->buffer, "Server: Cherokee/" VERSION CRLF, 17+6+2);
		}
	}

	/* Date
	 */
	cherokee_buffer_add (cnt->buffer, "Date: ", 6);
	cherokee_buffer_add_buffer (cnt->buffer, CONN_SRV(cnt)->bogo_now_string);
	cherokee_buffer_add (cnt->buffer, CRLF, 2);

	/* Authentication
	 */
	if ((cnt->realm_ref != NULL)  && 
	    (cnt->error_code == http_unauthorized)) 
	{
		cherokee_buffer_add (cnt->buffer, "WWW-Authenticate: Basic realm=\"", 31);
		cherokee_buffer_add_buffer (cnt->buffer, cnt->realm_ref);	
		cherokee_buffer_add (cnt->buffer, "\"" CRLF, 3);	
	}

	/* Redirected connections
	 */
	if (cnt->redirect->len >= 1) {
		cherokee_buffer_add (cnt->buffer, "Location: ", 10);
		cherokee_buffer_add (cnt->buffer, cnt->redirect->buf, cnt->redirect->len);
		cherokee_buffer_add (cnt->buffer, CRLF, 2);
	}

	/* Encoder headers
	 */
	if (cnt->encoder) {
		cherokee_encoder_add_headers (cnt->encoder, cnt->buffer);
		
		/* Keep-alive is not possible w/o a file cache
		 */
		cnt->keepalive = 0;

		if (cnt->handler->support & hsupport_length) {
			cnt->handler->support ^= hsupport_length;
		}
	}

	/* Unusual methods
	 */
	if (CONN_HDR(cnt)->method == http_options) {
		cherokee_buffer_add (cnt->buffer, "Allow: GET, HEAD, POST, OPTIONS"CRLF, 33);
	}
	
	/* Add the "Connection:" header
	 */
	if (cnt->handler && (cnt->keepalive > 0)) {
		cherokee_buffer_add (cnt->buffer, "Connection: Keep-Alive"CRLF, 24);
		cherokee_buffer_add_buffer (cnt->buffer, CONN_SRV(cnt)->timeout_header);
	} else {
		cherokee_buffer_add (cnt->buffer, "Connection: close"CRLF, 19);
	}
	
	/* If it's an error page we've to add the content-type header
	 */
	if (! http_type_200(cnt->error_code)) {
		cherokee_buffer_add (cnt->buffer, "Content-Type: text/html"CRLF, 25);		

		/* Add no-cache headers
		 */
		cherokee_buffer_add (cnt->buffer, "Cache-Control: no-cache"CRLF, 25);		
		cherokee_buffer_add (cnt->buffer, "Pragma: no-cache"CRLF, 18);		
		cherokee_buffer_add (cnt->buffer, "P3P: CP=3DNOI NID CURa OUR NOR UNI"CRLF, 36);		
	}

	/* Maybe the handler add some headers from the handler
	 */
	cherokee_buffer_add_buffer (cnt->buffer, cnt->header_buffer);

	/* Add the response header ends
	 */
	cherokee_buffer_add (cnt->buffer, CRLF, 2);
}


ret_t 
cherokee_connection_build_header (cherokee_connection_t *cnt)
{
	ret_t ret;

	if (http_type_200(cnt->error_code)) {
		/* Sanity checks
		 */
		if (cnt->handler == NULL) return ret_error;

		/* Try to get the headers from the handler
		 */
		ret = cherokee_handler_add_headers (cnt->handler, cnt->header_buffer);
		if (ret == ret_eagain) return ret;

		/* Does it need to be processed?
		 */
		if (HANDLER_SUPPORT_COMPLEX_HEADERS(cnt->handler)) {
			process_handler_complex_headers (cnt);
		}
	}

	/* Add the server headers	
	 */
	build_response_header (cnt);

	return ret_ok;
}


ret_t
cherokee_connection_send_header (cherokee_connection_t *cnt)
{
	ret_t ret;

	/* Send it
	 */
	ret = send_buffer_unsafe (cnt);

	cherokee_buffer_make_empty  (cnt->buffer);

	return ret;
}


ret_t 
cherokee_connection_send_header_and_mmaped (cherokee_connection_t *cnt)
{
	ssize_t re;
	struct iovec bufs[2];

	/* 1.- Special case: There is not header to send
	 * It is becase it has been sent in a writev()
	 */
	if (cherokee_buffer_is_empty (cnt->buffer)) {
		re = write (SOCKET_FD(cnt->socket), cnt->mmaped, cnt->mmaped_len);
		if (re < 0) {
			if (errno == EAGAIN) {
				return ret_eagain;
			}

			cnt->keepalive = 0;
			return ret_error;
		}
		
		CONN_VSRV(cnt)->data.tx += re;		

		cnt->mmaped_len -= re;
		cnt->mmaped     += re;

		return (cnt->mmaped_len <= 0) ? ret_ok : ret_eagain;
	}


	/* 2.- There are header and mmaped content to send
	 */
	bufs[0].iov_base = cnt->buffer->buf;
	bufs[0].iov_len  = cnt->buffer->len;
	bufs[1].iov_base = cnt->mmaped;
	bufs[1].iov_len  = cnt->mmaped_len;

	re = writev (SOCKET_FD(cnt->socket), bufs, 2);
	if (re <= 0) {
		cnt->keepalive = 0;
		return ret_error;
	} 
	
	/* If writev() has not sent all data
	 */
	if (re < cnt->buffer->len + cnt->mmaped_len) {
		int offset;

		if (re <= cnt->buffer->len) {
			cherokee_buffer_move_to_begin (cnt->buffer, re);
			return ret_eagain;
		}
		
		offset = re - cnt->buffer->len;
		cnt->mmaped     += offset;
		cnt->mmaped_len -= offset;

		cherokee_buffer_make_empty (cnt->buffer);

		return ret_eagain;
	}

	/* Add to the virtual host data sent counter
	 */
	CONN_VSRV(cnt)->data.tx += re;

	return ret_ok;
}


ret_t
cherokee_connection_recv (cherokee_connection_t *cnt, cherokee_buffer_t *buffer, uint32_t *len)
{
	ret_t   ret;
	ssize_t readed;
	
	ret = cherokee_socket_read (cnt->socket, buffer, DEFAULT_RECV_SIZE, &readed);
	switch (ret) {
	case ret_ok:
		/* Add to the virtual host data sent counter
		 */
		CONN_VSRV(cnt)->data.rx += readed;
		*len = readed;
		return ret_ok;

	case ret_eagain:
		return ret_eagain;

	case ret_eof:
		return ret_eof;
	}

	return ret_error;
}


int
cherokee_connection_eoh (cherokee_connection_t *cnt, cherokee_buffer_t *buffer, int tail_len)
{
	int   tail;
	char *start;

	/* Too few information?
	 * len("GET / HTTP/1.0" CRLF CRLF) = 18
	 */
	if (buffer->len < 18) {
		return 0;
	}

	/* Look for the starting point
	 */
	tail  = (tail_len < buffer->len) ? tail_len : buffer->len;
	start = buffer->buf + (buffer->len - tail);

	/* It could be a partial header, or maybe a POST request
	 */
	return (strstr(start, CRLF CRLF) != NULL);
}


ret_t 
cherokee_connection_reading_check (cherokee_connection_t *cnt)
{
	/* Check for too long headers
	 */
	if (cnt->incoming_header->len > MAX_HEADER_LEN) {
		cnt->error_code = http_bad_request;
		return ret_error;
	}

	return ret_ok;
}


ret_t 
cherokee_connection_set_cork (cherokee_connection_t *cnt, int enable)
{
	int on = 0;
	int fd;

#ifdef HAVE_TCP_CORK
	fd = SOCKET_FD(cnt->socket);
	if (enable) {
		setsockopt(fd, IPPROTO_TCP, TCP_NODELAY,  &on, sizeof on);

		on = 1;
		setsockopt(fd, IPPROTO_TCP, TCP_CORK,  &on, sizeof on);
	} else {
		setsockopt(fd, IPPROTO_TCP, TCP_CORK,  &on, sizeof on);

		on = 1;
		setsockopt(fd, IPPROTO_TCP, TCP_NODELAY,  &on, sizeof on);
	}

	cnt->tcp_cork = enable;
#endif

	return ret_ok;
}


ret_t
cherokee_connection_send (cherokee_connection_t *cnt)
{
	ret_t ret;
	int   sent;

	// sent = write (SOCKET_FD(cnt->socket), cnt->buffer->buf, cnt->buffer->len);
	ret = cherokee_socket_write (cnt->socket, cnt->buffer, &sent);

	if (sent > 0) {
		if (sent == cnt->buffer->len) {
			cherokee_buffer_make_empty (cnt->buffer);
		} else {
			cherokee_buffer_move_to_begin (cnt->buffer, sent);
		}

		/* If this connection has a handler without content-length support
		 * it has to count the bytes sent
		 */
		if (HANDLER_SUPPORT_LENGTH(cnt->handler)) {
			cnt->range_end += sent;
		}

		/* Add to the virtual host data sent counter
		 */
		CONN_VSRV(cnt)->data.tx += sent;

		return ret_ok;		
	}
	
	if (sent == 0) {
		return ret_eof;
	}

	if (sent < 0) {
		switch (errno) {
		case EAGAIN:
			return ret_eagain;
		default:
			/* Remote machine cancel the download
			 */
			return ret_error;
		}
	}
}


ret_t
cherokee_connection_close (cherokee_connection_t *cnt)
{
	return cherokee_socket_close (cnt->socket);
}


ret_t
cherokee_connection_step (cherokee_connection_t *cnt)
{
	ret_t ret = ret_ok;

	return_if_fail (cnt->handler != NULL, ret_error);

	/* Need to 'read' from handler ?
	 */
	if (! cherokee_buffer_is_empty (cnt->buffer)) {
		return ret_ok;
	}

	/* Do a step in the handler
	 */
	ret = cherokee_handler_step (cnt->handler, cnt->buffer);
	if ((ret != ret_ok) && (ret != ret_eof_have_data)) return ret;

	/* May be encode..
	 */
	if (cnt->encoder != NULL) {
		ret_t ret2;
		cherokee_buffer_t *tmp;

		/* Encode
		 */
		ret2 = cherokee_encoder_encode (cnt->encoder, cnt->buffer, cnt->encoder_buffer);
		if (ret2 < ret_ok) return ret2;

		/* Swap buffers	
		 */
		tmp = cnt->buffer;
		cnt->buffer = cnt->encoder_buffer;
		cnt->encoder_buffer = tmp;
	}
	
	return ret;
}


static inline ret_t
get_host (cherokee_connection_t *cnt, 
	  char                  *ptr) 
{
	/* ptr - Header at the "Host:" position 
	 */
	char *end, *end2;
	int   size;

	end = strchr (ptr, '\r');
	if (end == NULL) {
		return ret_error;
	}

	/* Drop the port if present
	 * Eg: www.alobbs.com:8080 -> www.alobbs.com
	 */ 
	end2 = strchr (ptr, ':');
	if (end2 && (end2 < end))  {
		end = end2;
	}

	size = (end - ptr);
	return cherokee_buffer_add (cnt->host, ptr, size);
}

static inline ret_t
get_encoding (cherokee_connection_t    *cnt,
	      char                     *ptr,
	      cherokee_encoder_table_t *encoders) 
{
	char tmp;
	char *i1, *i2;
	char *end;
	char *ext;

	/* ptr = Header at the "Accept-Encoding position 
	 */
	end = strchr (ptr, '\r');
	if (end == NULL) {
		return ret_error;
	}

	/* Look for the request extension
	 */
	ext = rindex (cnt->request->buf, '.');
	if (ext == NULL) {
		return ret_ok;
	}

	*end = '\0'; /* (1) */
	
	i1 = ptr;
	
	do {
		i2 = index (i1, ',');
		if (!i2) i2 = index (i1, ';');
		if (!i2) i2 = end;

		tmp = *i2;    /* (2) */

		*i2 = '\0';
		cherokee_encoder_table_new_encoder (encoders, i1, ext+1, &cnt->encoder);
		*i2 = tmp;    /* (2') */

		if (cnt->encoder != NULL) {
			/* Init the encoder related objects
			 */
			cherokee_encoder_init (cnt->encoder);
			cherokee_buffer_new (&cnt->encoder_buffer);
			break;
		}

		if (i2 < end) {
			i1 = i2+1;
		}

	} while (i2 < end);

	*end = '\r'; /* (1') */

	return ret_ok;
}


static inline ret_t
get_authorization (cherokee_connection_t *cnt,
		   char                  *ptr)
{
	cherokee_buffer_t *auth = NULL;

	if (strncasecmp(ptr, "Basic ", 6) == 0) {
		char *end;

		ptr += 6;
		end = strchr (ptr, '\r');
		if (end == NULL) {
			goto error;
		}

		cherokee_buffer_new (&auth);
		cherokee_buffer_add (auth, ptr, end-ptr+1);
		cherokee_buffer_decode_base64 (auth);

		end = strchr (auth->buf, ':');
		if (end == NULL) {
			goto error;
		}
		
		cherokee_buffer_add (cnt->user, auth->buf, end - auth->buf);
		cherokee_buffer_add (cnt->passwd, end+1, auth->len  - (end - auth->buf));		
	}

	return ret_ok;

error:
	if (auth != NULL) {
		cherokee_buffer_free (auth);
	}

	return ret_error;
}


int inline
cherokee_connection_is_userdir (cherokee_connection_t *cnt)
{
	return ((cnt->request->len > 4) && (cnt->request->buf[1] == '~'));
}


ret_t
cherokee_connection_build_local_directory (cherokee_connection_t *cnt, cherokee_virtual_server_t *vsrv, cherokee_handler_table_entry_t *entry)
{
	ret_t ret;

	if (entry->document_root->len >= 1) {
		/* Have a special DocumentRoot
		 */
		ret = cherokee_buffer_add_buffer (cnt->local_directory, entry->document_root);
		
		/* It has to drop the webdir from the request:
		 *
		 * Directory /thing {
		 *    DocumentRoot /usr/share/this/rocks
		 * }
		 *
		 * on petition: http://server/thing/cherokee
		 * should read: /usr/share/this/rocks/cherokee
		 */
		cherokee_buffer_move_to_begin (cnt->request, cnt->web_directory->len - 1);
		
	} else {

		/* Normal request
		 */
		ret = cherokee_buffer_add (cnt->local_directory, vsrv->root, vsrv->root_len);
	}

	return ret;
}


ret_t
cherokee_connection_build_local_directory_userdir (cherokee_connection_t *cnt, char *userdir)
{
	struct passwd *pwd;	
	char          *begin, *end_username;

	/* Find user name endding:
	 */
	begin = &cnt->request->buf[2];

	end_username = index (begin, '/');
	if (end_username == NULL) {
		end_username = &cnt->request->buf[cnt->request->len];
	}

	/* Get the user home directory
	 */
	cherokee_buffer_add (cnt->userdir, begin, end_username - begin);

	pwd = (struct passwd *) getpwnam (cnt->userdir->buf);
	if ((pwd == NULL) || (pwd->pw_dir == NULL)) {
		cnt->error_code = http_not_found;
		return ret_error;
	}

	/* Build the local_directory:
	 */
	cherokee_buffer_add (cnt->local_directory, pwd->pw_dir, strlen(pwd->pw_dir));
	cherokee_buffer_add (cnt->local_directory, "/", 1);
	cherokee_buffer_add (cnt->local_directory, userdir, strlen(userdir));

	/* Drop username from the request
	 */
	cherokee_buffer_move_to_begin (cnt->request, (end_username - cnt->request->buf));

	return ret_ok;
}


static inline ret_t
get_range (cherokee_connection_t *cnt, 
	   char                  *ptr) /* Header at the "Range: bytes=" position */
{
	int num_len = 0;

	/* Read the start position
	 */
	while ((ptr[num_len] != '-') && (ptr[num_len] != '\0') && (num_len < gbl_buffer_size-1)) {
		gbl_buffer[num_len] = ptr[num_len];
		num_len++;
	}
	gbl_buffer[num_len] = '\0';
	cnt->range_start = atoi (gbl_buffer);
	
	/* Advance the pointer
	 */
	ptr += num_len;
	if (*ptr == '-') {
		ptr++;
	} else {
		return ret_error;
	}

	/* Maybe there're an ending position
	 */
	if ((*ptr != '\0') && (*ptr != '\r') && (*ptr != '\n')) {
		num_len = 0;
		
		/* Read the end
		 */
		while ((ptr[num_len] != '-') && (ptr[num_len] != '\0') && (num_len < gbl_buffer_size-1)) {
			gbl_buffer[num_len] = ptr[num_len];
			num_len++;
		}
		gbl_buffer[num_len] = '\0';
		cnt->range_end = atoi (gbl_buffer);
	}

	return ret_ok;
}


static ret_t
post_init (cherokee_connection_t *cnt)
{
	int   r;
	ret_t ret;

	if (cnt->post != NULL) {
		SHOULDNT_HAPPEN;
	}

	cherokee_buffer_new (&cnt->post);

	/* Get the header "Content-Lenght" content
	 */
	ret = cherokee_header_copy_known (cnt->header, header_content_length, cnt->post);
	if (ret != ret_ok) {
		return ret_error;
	}

	/* Read the number
	 */
	r = sscanf (cnt->post->buf, "%d", &cnt->post_len);
	if (r <= 0) {
		return ret_error;
	}

	/* Clear the buffer to receive the post data
	 */
	cherokee_buffer_make_empty (cnt->post);
	
	return ret_ok;
}


ret_t 
cherokee_connection_get_request (cherokee_connection_t *cnt)
{
	ret_t  ret;

	/* Header parsing
	 */
	ret = cherokee_header_parse (cnt->header, cnt->incoming_header, header_type_request);
	if (ret < ret_ok) goto error;

	/* Maybe read the POST data
	 */
	if (HDR_METHOD(cnt->header) == http_post) {
		char *end;
		int   len;

		ret = post_init (cnt);
		if (ret != ret_ok) {
			/* TODO: Is this ok?
			 * Is ok a POST request w/o content-lenght?
			 */
			cnt->error_code = http_bad_request;
			return ret;
		}

/* TODO !!!
 * rewrite using header::_get_length
 */
		end = strstr(cnt->incoming_header->buf, CRLF CRLF);
		if (end == NULL) {
			cnt->error_code = http_bad_request;
			return ret_error;
		}
		
		end += 4;

		len = (cnt->incoming_header->buf + cnt->incoming_header->len) - end;
		cherokee_buffer_add (cnt->post, end, len);
		cherokee_buffer_drop_endding (cnt->incoming_header, len);
	}

	/* Copy the request
	 */
	ret = cherokee_header_copy_request (cnt->header, cnt->request);
	if (ret < ret_ok) goto error;

	/* Remove the "./" substring
	 */
	cherokee_buffer_remove_string (cnt->request, "./", 2);

	/* Look for ".."
	 */
	if (strstr (cnt->request->buf, "..") != NULL) {
		goto error;
	}

	/* Look for starting '/' in the request
	 */
	if (cnt->request->buf[0] != '/') {
		goto error;
	}

	/* Look for "//" 
	 */
	cherokee_buffer_remove_dups (cnt->request, '/');

	/* Check Host header in HTTP/1.1
	 */
	if (cnt->header->version == http_version_11) {
		char *host;
		int   host_len;

		if (cherokee_header_get_known (cnt->header, header_host, 
					       &host, &host_len) == ret_not_found) 
		{
			goto error;
		}

		get_host (cnt, host);
	}

	cnt->error_code = http_ok;
	return ret_ok;	

error:
	cnt->error_code = http_bad_request;
	return ret_error;
}


ret_t 
cherokee_connection_get_plugin_entry (cherokee_connection_t *cnt, cherokee_handler_table_t *plugins, cherokee_handler_table_entry_t **plugin_entry)
{
	ret_t ret;

	return_if_fail (plugins != NULL, ret_error);

	/* Look for the handler "*_new" function
	 */
	ret = cherokee_handler_table_get (plugins, cnt->request->buf, plugin_entry, cnt->web_directory);
	if (ret != ret_ok) {
		cnt->error_code = http_internal_error;
		return ret_error;
	}	

	/* Set the referece to the realm
	 */
	cnt->realm_ref = (*plugin_entry)->auth_realm;

	return ret_ok;
}


ret_t 
cherokee_connection_check_authentication (cherokee_connection_t *cnt, cherokee_handler_table_entry_t *plugin_entry)
{
	if (plugin_entry->validator_new_func != NULL) {
		ret_t ret;
		char *ptr;
		int   len;
		cherokee_validator_t *validator = NULL;

		/* Look for authentication in the headers:
		 * It's done on demand because the directory maybe don't have protection
		 */
		ret = cherokee_header_get_unknown (cnt->header, "Authorization", 13, &ptr, &len);
		if (ret < ret_ok) {
			goto unauthorized;
		}

		ret = get_authorization (cnt, ptr);
		if (ret != ret_ok) {
			goto unauthorized;
		}
		
		/* Create the validator object
		 */
		ret = plugin_entry->validator_new_func ((void **)&validator, 
							plugin_entry->properties);
		if (ret != ret_ok) {
			cnt->error_code = http_internal_error;
			return ret_error;	
		}

		/* Check the login/password
		 */
		ret = cherokee_validator_check (validator, cnt);
		cherokee_validator_free (validator);

		if (ret != ret_ok) {
			goto unauthorized;
		}
       	}

	return ret_ok;

unauthorized:
	cnt->keepalive = 0;
	cnt->error_code = http_unauthorized;
	return ret_error;
}


ret_t 
cherokee_connection_check_ip_validation (cherokee_connection_t *cnt, cherokee_handler_table_entry_t *plugin_entry)
{
	if (plugin_entry->access != NULL) {
		ret_t ret;

		ret = cherokee_access_ip_match (plugin_entry->access, cnt->socket);
		if (ret == ret_not_found) {
			return ret_ok;
		}

		cnt->error_code = http_access_denied;
		return ret_error;
	}
	return ret_ok;
}


ret_t 
cherokee_connection_create_handler (cherokee_connection_t *cnt, cherokee_handler_table_entry_t *plugin_entry)
{
	ret_t ret;

	return_if_fail (plugin_entry->handler_new_func != NULL, ret_error);

	/* Create and assign a handler object
	 */
	ret = (plugin_entry->handler_new_func) ((void **)&cnt->handler, cnt, plugin_entry->properties);
	if ((ret != ret_ok) || (cnt->handler == NULL)) {
		cnt->error_code = http_internal_error;
		return ret_error;
	}

	return ret_ok;
}


ret_t
cherokee_connection_parse_header (cherokee_connection_t *cnt, cherokee_encoder_table_t *encoders)
{	   
	ret_t  ret;
	char  *ptr;
	int    ptr_len;

	/* Look for "Connection: Keep-Alive / Close"
	 */
	ret = cherokee_header_get_known (cnt->header, header_connection, &ptr, &ptr_len);
	if (ret == ret_ok) 
	{
		if (strncasecmp (ptr, "close", 5) == 0) {
			cnt->keepalive = 0;
		}

	} else {
		cnt->keepalive = 0;
	}

	/* Look for "Range:" 
	 */
	if (HANDLER_SUPPORT_RANGE(cnt->handler)) {
		ret = cherokee_header_get_known (cnt->header, header_range, &ptr, &ptr_len);
		if (ret == ret_ok) {
			if (strncmp (ptr, "bytes=", 6) == 0) {
				ptr += 6;
				get_range (cnt, ptr);
			}
		}
	}

	/* Look for "Accept-Encoding:"
	 */
	ret = cherokee_header_get_known (cnt->header, header_accept_encoding, &ptr, &ptr_len);
	if (ret == ret_ok) {
		get_encoding (cnt, ptr, encoders);
	}

	return ret_ok;
}


ret_t 
cherokee_connection_parse_args (cherokee_connection_t *cnt)
{
	ret_t ret;

	ret = cherokee_table_new (&cnt->arguments);
	if (ret < ret_ok) return ret;

	return cherokee_header_get_arguments (cnt->header, cnt->query_string, cnt->arguments);
}


ret_t
cherokee_connection_open_request (cherokee_connection_t *cnt)
{	
#if 0
	printf ("cherokee_connection_open_request:\n\tweb_dir '%s'\n\trequest '%s'\n\tlocal   '%s'\n", 
		cnt->web_directory->buf,
		cnt->request->buf,
		cnt->local_directory->buf);
#endif

	/* If the connection is keep-alive, have a look at the
	 * handler to see if supports it.
	 */
	if (HANDLER_SUPPORT_LENGTH(cnt->handler) == 0) {
		cnt->keepalive = 0;
	}
	
	/* Ensure the space for writting
	 */
	cherokee_buffer_ensure_size (cnt->buffer, DEFAULT_READ_SIZE);

	return cherokee_handler_init (cnt->handler);
}


ret_t
cherokee_connection_send_response_page_hardcoded (cherokee_connection_t *cnt)
{
	   ret_t  ret;
	   
	   cherokee_buffer_make_empty  (cnt->buffer);
	   
	   cherokee_buffer_add (cnt->buffer, "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">" CRLF, 50);
	   
	   /* Add page title
	    */
	   cherokee_buffer_add (cnt->buffer, "<html><head><title>", 19);
	   add_error_code_string_to_buffer (cnt);

	   /* Add big banner
	    */
	   cherokee_buffer_add (cnt->buffer, "</title></head><body><h1>", 25);
	   add_error_code_string_to_buffer (cnt);
	   cherokee_buffer_add (cnt->buffer, "</h1>", 5);

	   /* Maybe add some info
	    */
	   switch (cnt->error_code) {
	   case http_not_found:
		   if (cnt->request) {
			   cherokee_buffer_swap_chars (cnt->request, '<', ' ');
			   cherokee_buffer_swap_chars (cnt->request, '>', ' ');

			   cherokee_buffer_add (cnt->buffer, "The requested URL ", 18);
			   cherokee_buffer_add_buffer (cnt->buffer, cnt->request);
			   cherokee_buffer_add (cnt->buffer, " was not found on this server.", 30);
		   }
		   break;
	   case http_bad_request:
		   cherokee_buffer_add (cnt->buffer, 
					"Your browser sent a request that this server could not understand.", 66);
		   break;
	   case http_moved_permanently:
		   cherokee_buffer_add (cnt->buffer, 
					"The document has moved", 22);
		   break;
	   case http_unauthorized:
		   cherokee_buffer_add (cnt->buffer, 
					"This server could not verify that you are authorized to access the document "
					"requested.  Either you supplied the wrong credentials (e.g., bad password), "
					"or your browser doesn't understand how to supply the credentials required", 225);
	   default:
		   break;
	   }
	   
	   /* Add page foot
	    */
	   if (CONN_SRV(cnt)->hideservername == 0) {
		   cherokee_buffer_add (cnt->buffer, "<p><hr>", 7); 

		   if (CONN_SRV(cnt)->hideversion) {
			   cherokee_buffer_add_version (cnt->buffer, CONN_SRV(cnt)->port, ver_port_html);
		   } else {
			   cherokee_buffer_add_version (cnt->buffer, CONN_SRV(cnt)->port, ver_full_html);
		   }
	   }
  
	   cherokee_buffer_add (cnt->buffer, "</body></html>", 14); 
 
	   /* Send it
	    */
	   return send_buffer_unsafe (cnt);
}


ret_t 
cherokee_connection_send_response_page_file (cherokee_connection_t *cnt, char *filename)
{
	cherokee_buffer_make_empty  (cnt->buffer);
	cherokee_buffer_read_file (cnt->buffer, filename);

	return send_buffer_unsafe (cnt);
}


ret_t 
cherokee_connection_log_or_delay (cherokee_connection_t *conn)
{
	ret_t ret = ret_ok;
	
	if (conn->handler == NULL) {
		conn->log_at_end = 0;
	} else {
		conn->log_at_end = ! HANDLER_SUPPORT_LENGTH(conn->handler);
	}

	if (conn->log_at_end == 0) {
		if (conn->logger_ref != NULL) {
			ret = cherokee_logger_write_access (conn->logger_ref, conn);
		}
	}

	return ret;
}


ret_t 
cherokee_connection_log_delayed (cherokee_connection_t *conn)
{
	ret_t ret = ret_ok;

	if (conn->log_at_end) {
		if (conn->logger_ref != NULL) {
			ret = cherokee_logger_write_access (conn->logger_ref, conn);
		}		
	}

	return ret_ok;
}
