/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 * 
 * This piece of code by:
 *      Pablo Neira Ayuso <pneira@optimat.com>
 *      Miguel Angel Ajo Pelayo <ajo@godsmaze.org> 
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 * 
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <sys/types.h>
#include <unistd.h>
#include <syslog.h>

#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else 
#include <time.h>
#endif

#include "logger_ncsa.h"
#include "connection.h"
#include "server.h"
#include "module.h"


/* Some constants
 */
static char *method[]  = {"GET", "POST", "HEAD", "UNKNOWN", NULL};
static char *version[] = {"HTTP/0.9", "HTTP/1.0", "HTTP/1.1", "UNKNOWN", NULL};
static char *month[]   = {"Jan", "Feb", "Mar", "Apr", "May", "Jun",
			 "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", NULL};

#ifdef HAVE_IPV6
#define IN_ADDR(c) ((struct in6_addr)(c).sin6_addr)
#define IN_AF(c)   ((c).sin6_family)
#else
#define IN_ADDR(c) ((struct in_addr)(c).sin_addr)
#define IN_AF(c)   ((c).sin_family)
#endif


cherokee_module_info_t cherokee_ncsa_info = {
	cherokee_logger,            /* type     */
	cherokee_logger_ncsa_new    /* new func */
};

ret_t
cherokee_logger_ncsa_new  (cherokee_logger_t **logger, cherokee_table_t *properties)
{
	CHEROKEE_NEW_STRUCT (n, logger_ncsa);
	
	/* Init the base class object
	 */
	cherokee_logger_init_base(LOGGER(n));

	LOGGER(n)->init         = (logger_func_init_t) cherokee_logger_ncsa_init;
	LOGGER(n)->free         = (logger_func_free_t) cherokee_logger_ncsa_free;
	LOGGER(n)->flush        = (logger_func_flush_t) cherokee_logger_ncsa_flush;
	LOGGER(n)->write_error  = (logger_func_write_error_t)  cherokee_logger_ncsa_write_error;
	LOGGER(n)->write_access = (logger_func_write_access_t) cherokee_logger_ncsa_write_access;
	LOGGER(n)->write_string = (logger_func_write_string_t) cherokee_logger_ncsa_write_string;

	*logger = LOGGER(n);
	
	/* Init
	 */
	n->errorlog_fd        = NULL;
	n->accesslog_fd       = NULL;
	n->accesslog_filename = NULL;
	n->errorlog_filename  = NULL;

	if (properties != NULL) {
		n->accesslog_filename = cherokee_table_get_val (properties, "AccessLog");
		n->errorlog_filename  = cherokee_table_get_val (properties, "ErrorLog");
	}
	
	return ret_ok;
}


ret_t 
cherokee_logger_ncsa_init (cherokee_logger_ncsa_t *logger)
{
	if ((logger->accesslog_filename == NULL) ||
	    (logger->errorlog_filename == NULL))
	{
		openlog ("Cherokee", LOG_CONS | LOG_PID | LOG_NDELAY, LOG_LOCAL1);
		return ret_ok;
	}

	logger->accesslog_fd = fopen (logger->accesslog_filename, "a+");
	if (logger->accesslog_fd == NULL) {
		PRINT_ERROR("cherokee_logger_ncsa: error opening %s for append\n", logger->accesslog_filename); 
		return ret_error;
	}
	fcntl (fileno (logger->accesslog_fd), F_SETFD, 1);
	

        logger->errorlog_fd  = fopen (logger->errorlog_filename, "a+");
	if (logger->errorlog_fd == NULL) {
		PRINT_ERROR("cherokee_logger_ncsa: error opening %s for append\n", logger->errorlog_filename); 
		return ret_error;
	}
	fcntl (fileno (logger->errorlog_fd), F_SETFD, 1);

	return ret_ok;
}


ret_t
cherokee_logger_ncsa_free (cherokee_logger_ncsa_t *logger)
{
	int ret, n = 2;

	if (logger->errorlog_fd != NULL) {
		ret = fclose (logger->errorlog_fd);
		n--;
	}
	
	if (logger->accesslog_fd != NULL) {
		ret |= fclose (logger->errorlog_fd);
		n--;
	}
	
	if (n != 0) {
		closelog();
	}
	
	free (logger);
	
	return (ret == 0) ? ret_ok : ret_error;
}


ret_t 
cherokee_logger_ncsa_flush (cherokee_logger_ncsa_t *logger)
{
	int   tmp;
	ret_t ret = ret_ok;

	if (cherokee_buffer_is_empty (LOGGER_BUFFER(logger))) {
		return ret_ok;
	}

	if (logger->accesslog_fd == NULL) {
		syslog (LOG_INFO, "%s", LOGGER_BUFFER(logger)->buf);
		return cherokee_buffer_make_empty (LOGGER_BUFFER(logger));
	}


	tmp = fwrite (LOGGER_BUFFER(logger)->buf, 1, LOGGER_BUFFER(logger)->len, logger->accesslog_fd); 
	fflush (logger->accesslog_fd);
	if (tmp < 0) {
		return ret_error;
	}

	if (tmp == LOGGER_BUFFER(logger)->len) {
		return cherokee_buffer_make_empty (LOGGER_BUFFER(logger));
	} 

	return cherokee_buffer_drop_endding (LOGGER_BUFFER(logger), tmp);
}


ret_t 
cherokee_logger_ncsa_write_error  (cherokee_logger_ncsa_t *logger, cherokee_connection_t *cnt)
{
	long int z;
	int len;
	struct tm conn_time;
	char ipaddr[INET6_ADDRSTRLEN+1];
	CHEROKEE_TEMP(tmp, 200);

	localtime_r (&CONN_SRV(cnt)->bogo_now, &conn_time);

#ifdef HAVE_INT_TIMEZONE
	z = - (timezone / 60);
#else
#warning TODO
	z = 0;
#endif
	
	memset(ipaddr, 0, sizeof(ipaddr));
	cherokee_socket_ntop(cnt->socket, ipaddr, sizeof(ipaddr)-1);

	len = snprintf (tmp, tmp_size-1,
			"%s - - [%02d/%s/%d:%02d:%02d:%02d %c%02d%02d] [error] \"%s %s %s\"\n",
			ipaddr,
			conn_time.tm_mday, 
			month[conn_time.tm_mon], 
			1900 + conn_time.tm_year,
			conn_time.tm_hour, 
			conn_time.tm_min, 
			conn_time.tm_sec,
			(z < 0) ? '-' : '+', 
			(int) z/60, 
			(int) z%60, 
			method[cnt->header->method],
			cnt->request->buf, 
			version[cnt->header->version]);


	if (logger->errorlog_fd != NULL) {
		size_t wrote;

		if ((len > tmp_size-1) || (len == -1)) {
			len = tmp_size;
			tmp[tmp_size-1] = '\n';
		}

		wrote = fwrite (tmp, 1, len, logger->errorlog_fd); 
                fflush(logger->errorlog_fd);

		return (wrote > 0) ? ret_ok : ret_error;
	}

	syslog (LOG_ERR, "%s", tmp);
	return ret_ok;
}


ret_t 
cherokee_logger_ncsa_write_string (cherokee_logger_ncsa_t *logger, const char *string)
{
	if (logger->accesslog_fd != NULL) {
		int ret;
		ret = fprintf (logger->accesslog_fd, string);

		return (ret > 0) ? ret_ok : ret_error;
	} 

	syslog (LOG_INFO, "%s", string);
	return ret_ok;
}


ret_t
cherokee_logger_ncsa_write_access (cherokee_logger_ncsa_t *logger, cherokee_connection_t *cnt)
{
	ret_t ret = ret_ok;
  	long int z;
	int  len;
	struct tm conn_time;
	char ipaddr[INET6_ADDRSTRLEN];
	CHEROKEE_TEMP (tmp, 400);
    
	/* thread-safe version of localtime 
	 */
	localtime_r(&CONN_SRV(cnt)->bogo_now, &conn_time);
	
#ifdef HAVE_INT_TIMEZONE
	z = - (timezone / 60);
#else
#warning TODO
	z = 0;
#endif

	memset(ipaddr, 0, sizeof(ipaddr));
	cherokee_socket_ntop(cnt->socket, ipaddr, sizeof(ipaddr)-1);

	len = snprintf (tmp, tmp_size,
			"%s - - [%02d/%s/%d:%02d:%02d:%02d %c%02d%02d] \"%s %s %s\" %d %ld\n",
			ipaddr,
			conn_time.tm_mday, 
			month[conn_time.tm_mon], 
			1900 + conn_time.tm_year,
			conn_time.tm_hour, 
			conn_time.tm_min, 
			conn_time.tm_sec,
			(z < 0) ? '-' : '+', 
			(int) z/60, 
			(int) z%60, 
			method[cnt->header->method],
			cnt->request->buf, 
			version[cnt->header->version], 
			cnt->error_code,
			(long) cnt->range_end - cnt->range_start);
	
	ret = cherokee_buffer_add (LOGGER_BUFFER(logger), tmp, len);

	return ret;
}




/*   Library init function
 */

static int _ncsa_is_init = 0;

void
ncsa_init ()
{
	/* Init flag
	 */
	if (_ncsa_is_init) {
		return;
	}
	_ncsa_is_init = 1;
}


