/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/* Cherokee
 *
 * Authors:
 *      Alvaro Lopez Ortega <alvaro@alobbs.com>
 *
 * This piece of code by:
 *      Ricardo Cardenes Medina <ricardo@conysis.com>
 *
 * Copyright (C) 2001, 2002, 2003, 2004 Alvaro Lopez Ortega
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "common.h"

#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#ifdef HAVE_NETINET_TCP_H
# include <netinet/tcp.h>
#endif

#ifdef HAVE_SYS_FILIO_H
# include <sys/filio.h>     /* defines FIONBIO and FIONREAD */
#endif
#ifdef HAVE_SYS_SOCKIO_H
# include <sys/sockio.h>    /* defines SIOCATMARK */
#endif

#ifdef HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif

#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include "buffer.h"
#include "socket.h"
#include "virtual_server.h"


#ifdef HAVE_IPV6
 typedef struct sockaddr_in6 real_socket_t;
#else
 typedef struct sockaddr_in real_socket_t;
#endif

#define N_BITS 512


ret_t
cherokee_socket_new (cherokee_socket_t **socket)
{
	CHEROKEE_NEW_STRUCT (n, socket);

	/* Init 
	 */
	n->socket   = -1;
	n->addr_len = sizeof(n->addr_in);
	n->status   = socket_closed;
	n->is_tls   = 0;

#ifdef HAVE_TLS
	n->initialized = 0;
#endif

	/* Return it
	 */
	*socket = n;

	return ret_ok;
}


#ifdef HAVE_TLS

/*
 * This function gets called by GnuTLS to determine which certificate
 * to use.
 *
 * It checks if the client indicated which host it is connecting to,
 * and if yes, tries to find an appropriate cert.
 */
static int
session_cert_select (gnutls_session  session,
		     gnutls_datum   *server_certs, 
		     int             ncerts)
{
	int i;
	int name_type;
	int data_length = 0;

	/* Get the length of the data. Note that we only care about the
	 * first hostname sent. At least, for now. 
	 */
	data_length = 0;
	i = gnutls_server_name_get (session, NULL, &data_length, &name_type, 0);

	printf ("intenta conectar con '%s'\n");

	return 0;
}


static ret_t
initialize_tls_session (cherokee_socket_t *socket, cherokee_virtual_server_t *vserver)
{
/*
	gnutls_session session;
	const int cert_type_prio[] = { GNUTLS_CRT_OPENPGP, 0 };
	const int comp_prio[]      = { GNUTLS_COMP_ZLIB, GNUTLS_COMP_LZO,
				       GNUTLS_COMP_NULL, 0 };
	const int mac_prio[]       = { GNUTLS_MAC_SHA, GNUTLS_MAC_MD5, 0 };
	const int kx_prio[]        = { GNUTLS_KX_DHE_DSS, GNUTLS_KX_RSA,
				       GNUTLS_KX_DHE_RSA, GNUTLS_KX_SRP,
				       GNUTLS_KX_RSA_EXPORT, 0 };
	const int cipher_prio[]    = { GNUTLS_CIPHER_RIJNDAEL_CBC, 
				       GNUTLS_CIPHER_3DES_CBC,
				       GNUTLS_CIPHER_ARCFOUR_128, 
				       GNUTLS_CIPHER_ARCFOUR_40, 0 };
	const int protocol_prio[]  = { GNUTLS_TLS1, GNUTLS_SSL3, 0 };
*/
  
	gnutls_init (&socket->session, GNUTLS_SERVER);

	/* Use the default priorities, plus, export cipher suites.
	 */
	gnutls_set_default_export_priority(socket->session);
	
	/* Avoid calling all the priority functions, since the defaults
	 * are adequate.
	 */
//	gnutls_set_default_priority (socket->session);   

/*
	gnutls_cipher_set_priority (socket->session, cipher_prio);
	gnutls_compression_set_priority (socket->session, comp_prio);
	gnutls_kx_set_priority (socket->session, kx_prio);
	gnutls_protocol_set_priority (socket->session, protocol_prio);
	gnutls_mac_set_priority (socket->session, mac_prio);
*/

/*	
	if (cert_type == THY_TLS_CERT_TYPE_OPENPGP) {
		gnutls_certificate_type_set_priority (socket->session, cert_type_prio);
	}
*/	
	gnutls_credentials_set (socket->session, GNUTLS_CRD_CERTIFICATE, vserver->credentials);
/*
	if (cert_type == THY_TLS_CERT_TYPE_OPENPGP) {
		gnutls_certificate_server_set_request (socket->session, GNUTLS_CERT_REQUEST);
	}
*/
//	gnutls_handshake_set_rsa_pms_check (socket->session, 1);
  
//	gnutls_db_set_retrieve_function (socket->session, thy_tls_db_fetch);
//	gnutls_db_set_remove_function (socket->session, thy_tls_db_delete);
//	gnutls_db_set_store_function (socket->session, thy_tls_db_store);
//	gnutls_db_set_ptr (socket->session, NULL);

	/* Request client certificate if any.	
	 */
	gnutls_certificate_server_set_request (socket->session, GNUTLS_CERT_REQUEST);

	gnutls_dh_set_prime_bits (socket->session, 1024);


//	gnutls_handshake_set_private_extensions (socket->session, 1);

//	if (cert_type == THY_TLS_CERT_TYPE_X509) {
//		gnutls_certificate_server_set_select_function (
//			socket->session, 
//			(gnutls_certificate_server_select_function *) session_cert_select);
//	}

	return ret_ok;
}
#endif /* HAVE_TLS */


ret_t 
cherokee_socket_init_tls (cherokee_socket_t *socket, cherokee_virtual_server_t *vserver)
{
#ifdef HAVE_TLS
	int rc;

	if (socket->initialized == 0) {
		initialize_tls_session (socket, vserver);
		gnutls_transport_set_ptr (socket->session, socket->socket);
		socket->initialized = 1;
	}


	PRINT_DEBUG ("handshake: pre\n");
	rc = gnutls_handshake (socket->session);
	PRINT_DEBUG ("handshake: post %d\n", rc);

	switch (rc) {
	case GNUTLS_E_AGAIN:
		printf ("handshake eagin %d\n", rc);
		return ret_eagain;
		break;
	case GNUTLS_E_INTERRUPTED:
		printf ("handshake interrupted\n");
		return ret_error;
		break;
	case 0:
		printf ("handshake 0\n");
	}
	
	if (rc < 0) {
		PRINT_ERROR ("Init TLS: Handshake has failed: %s\n", gnutls_strerror(rc));
		gnutls_deinit (socket->session);
		return ret_error;
	}

	socket->is_tls = 1;
	printf ("Handshake was completed\n");
#endif

	return ret_ok;
}


ret_t
cherokee_socket_free (cherokee_socket_t *socket)
{
	free (socket);
	return ret_ok;
}

ret_t       
cherokee_socket_close (cherokee_socket_t *socket)
{
	int re;

	if (socket->socket <= 0) {
		return ret_error;
	}

#ifdef HAVE_TLS
	if (socket->is_tls) {
		gnutls_bye (socket->session, GNUTLS_SHUT_WR);
		gnutls_deinit (socket->session);
	}
#endif

	re = close (socket->socket);
	socket->socket = -1;
	socket->status = socket_closed;

	return (re == 0) ? ret_ok : ret_error;
}


const char *
cherokee_socket_ntop (cherokee_socket_t *socket, char *dst, size_t cnt)
{
	char *ret = NULL;
	errno = EAFNOSUPPORT;

	if (SOCKET_FD(socket) <= 0) {
		return NULL;
	}

#ifdef HAVE_INET_PTON
# ifdef HAVE_IPV6
	if (SOCKET_AF(socket) == AF_INET6) {
		/* We assume that inet_ntop is there too */
		ret = (char *) inet_ntop(AF_INET6,
					 &((struct sockaddr_in6 *)&SOCKET_ADDR(socket))->sin6_addr,
					 dst, cnt);
	}
# endif /* HAVE_IPV6 */
	if (SOCKET_AF(socket) == AF_INET) {
		ret = (char *) inet_ntop(AF_INET,
					 (void *) &((struct sockaddr_in *)&SOCKET_ADDR(socket))->sin_addr,
					 dst, cnt);
	}
#else /* !HAVE_INET_PTON */
	if (SOCKET_AF(socket) == AF_INET) {
		ret = inet_ntoa(((struct sockaddr_in *)SOCKET_ADDR(socket))->sin_addr);
		strncpy(dst, ret, cnt);
		ret = dst;
	}
#endif /* HAVE_INET_PTON */

	return ret;
}


ret_t
cherokee_socket_accept (cherokee_socket_t *socket, int server_socket)
{
	int tmp = 1;
	int new_socket;

	/* Get the new connection
	 */
	new_socket = accept (server_socket, (struct sockaddr *)&socket->addr_in, &socket->addr_len);
	if (new_socket <= 0) {
		return ret_error;
	}


	/* Set some parameters to the socket
	 */

	/* <EXPERIMENTAL>
	 */
	fcntl (new_socket, F_SETFD, 1);  /* close-on-exec */
	
	/* Disable Nagle's algorithm for this connection.
	 * Written data to the network is not buffered pending
	 * acknowledgement of previously written data.
	 */
 	setsockopt (new_socket, IPPROTO_TCP, TCP_NODELAY, &tmp, sizeof(tmp));

	/*  </EXPERIMENTAL>
	 */
	
	/* Enables nonblocking I/O.
	 */
	ioctl (new_socket, FIONBIO, &tmp);

	SOCKET_FD(socket) = new_socket;

	return ret_ok;
}


ret_t       
cherokee_socket_set_status (cherokee_socket_t *socket, cherokee_socket_status_t status)
{
	socket->status = status;

	if (status == socket_closed) {
		cherokee_socket_close (socket);
	}

	return ret_ok;
}


static ret_t 
cherokee_write (cherokee_socket_t *socket, const char *buf, int buf_len, int *writed)
{
	ssize_t rc;

#if HAVE_TLS

#else
	rc = write (SOCKET_FD(socket), buf, buf_len);
#endif
	
	if (rc < 0) {
		switch (errno) {
		case EAGAIN:     return ret_eagain;
		case EPIPE:      return ret_eof;
		case ECONNRESET: return ret_eof;
		}
	
		PRINT_ERROR ("Unknown error: write(%d, ..) -> errno=%d '%s'\n", 
			     SOCKET_FD(socket), errno, strerror(errno));
		return ret_error;
	}

	/* Return info
	 */
	*writed = rc;
	return ret_ok;
}


static ret_t
cherokee_read (cherokee_socket_t *socket, char *buf, int buf_size, ssize_t *readed)
{
#ifdef HAVE_TLS

#else	
	*readed = read (SOCKET_FD(socket), buf, buf_size);
#endif

	if (*readed < 0) {
		switch (errno) {
		case EAGAIN:     return ret_eagain;
		case EPIPE:      return ret_eof;
		case ECONNRESET: return ret_eof;
		}

		PRINT_ERROR ("Unknown error: read(%d, ..) -> errno=%d '%s'\n", 
			     SOCKET_FD(socket), errno, strerror(errno));
		return ret_error;

	} else if (*readed == 0) {
		return ret_eof;
	}
	
	return ret_ok;
}


ret_t       
cherokee_socket_write (cherokee_socket_t *socket, cherokee_buffer_t *buf, size_t *writed)
{
	return cherokee_write (socket, buf->buf, buf->len, writed);
}


ret_t      
cherokee_socket_read (cherokee_socket_t *socket, cherokee_buffer_t *buf, size_t count, ssize_t *readed)
{
	ret_t    ret;
	char    *starting;
	
	cherokee_buffer_ensure_size (buf, buf->len + count);
	starting = buf->buf + buf->len;

	ret = cherokee_read (socket, starting, count, readed);
	if (ret == ret_ok) {
		buf->len += *readed;
	}

	return ret;
}
