/*
 * Copyright (C) 1995, 1996, 1997, and 1998 WIDE Project.
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    This product includes software developed by WIDE Project and
 *    its contributors.
 * 4. Neither the name of the project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE PROJECT AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE PROJECT OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* YIPS @(#)$Id: pfkey_debug.c,v 1.1.1.1.2.3 1998/07/02 06:59:48 sakane Exp $ */

#if defined(YIPS_DEBUG)

#include <sys/types.h>
#include <sys/param.h>
#include <sys/mbuf.h>
#include <sys/socket.h>

#include <net/pfkeyv2.h>

#include <netinet/in.h>

#include "var.h"
#include "vmbuf.h"
#include "misc.h"
#include "pfkey.h"
#include "pfkey_lib.h"

/* NOTE: host byte order */

void kdebug_sadb __P((struct sadb_msg *));
void kdebug_sadb_sa __P((struct sadb_ext *));
void kdebug_sadb_address __P((struct sadb_ext *));
void kdebug_sadb_key __P((struct sadb_ext *));
void kdebug_sadb_policy __P((struct sadb_ext *));
void kdebug_vchar __P((vchar_t *));
void kdebug_pdump __P((caddr_t, int, int));
void kdebug_sockaddr __P((struct sockaddr *));

/* %%%: about struct sadb_msg */
void
kdebug_sadb(base)
	struct sadb_msg *base;
{
	struct sadb_ext *ext;
	int tlen;

	/* sanity check */
	if (base == 0)
		printf("kdebug_sadb: NULL pointer was passed.\n");

	printf("sadb_msg{ version=%u type=%u errno=%u satype=%u\n",
	    base->sadb_msg_version, base->sadb_msg_type,
	    base->sadb_msg_errno, base->sadb_msg_satype);
	printf("  len=%u ext=0x%04x seq=0x%08x pid=%u }\n",
	    base->sadb_msg_len, base->sadb_msg_reserved,
	    base->sadb_msg_seq, base->sadb_msg_pid);

	tlen = base->sadb_msg_len - sizeof(struct sadb_msg);
	ext = (struct sadb_ext *)((caddr_t)base + sizeof(struct sadb_msg));

	while (tlen > 0) {
		printf("sadb_ext{ len=%u type=%u }\n",
		    ext->sadb_ext_len, ext->sadb_ext_type);

		switch (ext->sadb_ext_type) {
		case SADB_EXT_SA:
			kdebug_sadb_sa(ext);
			break;
		case SADB_EXT_LIFETIME_CURRENT:
		case SADB_EXT_LIFETIME_HARD:
		case SADB_EXT_LIFETIME_SOFT:
			break;
		case SADB_EXT_ADDRESS_SRC:
		case SADB_EXT_ADDRESS_DST:
		case SADB_EXT_ADDRESS_PROXY:
			kdebug_sadb_address(ext);
			break;
		case SADB_EXT_KEY_AUTH:
		case SADB_EXT_KEY_ENCRYPT:
			kdebug_sadb_key(ext);
			break;
		case SADB_EXT_IDENTITY_SRC:
		case SADB_EXT_IDENTITY_DST:
		case SADB_EXT_SENSITIVITY:
		case SADB_EXT_PROPOSAL:
		case SADB_EXT_SUPPORTED_AUTH:
		case SADB_EXT_SUPPORTED_ENCRYPT:
		case SADB_EXT_SPIRANGE:
		case SADB_X_EXT_KMPRIVATE:
			break;
		case SADB_X_EXT_POLICY:
			kdebug_sadb_policy(ext);
			break;
		default:
			printf("kdebug_sadb: invalid ext_type %u was passed.\n",
			    ext->sadb_ext_type);
			return;
		}

		tlen -= ext->sadb_ext_len;
		ext = (struct sadb_ext *)((caddr_t)ext + ext->sadb_ext_len);
	}

	return;
}

void
kdebug_sadb_sa(ext)
	struct sadb_ext *ext;
{
	struct sadb_sa *sa = (struct sadb_sa *)ext;

	/* sanity check */
	if (ext == 0)
		printf("kdebug_sadb_sa: NULL pointer was passed.\n");

	printf("sadb_sa{ spi=0x%08x replay=%u state=%u\n",
	    sa->sadb_sa_spi, sa->sadb_sa_replay, sa->sadb_sa_state);
	printf("  auth=%u encrypt=%u flags=0x%08x }\n",
	    sa->sadb_sa_auth, sa->sadb_sa_encrypt, sa->sadb_sa_flags);

	return;
}

void
kdebug_sadb_address(ext)
	struct sadb_ext *ext;
{
	struct sadb_address *addr = (struct sadb_address *)ext;

	/* sanity check */
	if (ext == 0)
		printf("kdebug_sadb_address: NULL pointer was passed.\n");

	printf("sadb_address{ proto=%u prefixlen=%u reserved=%02x %02x }\n",
	    addr->sadb_address_proto, addr->sadb_address_prefixlen,
	    ((caddr_t)&addr->sadb_address_reserved)[0],
	    ((caddr_t)&addr->sadb_address_reserved)[1]);

	kdebug_sockaddr((struct sockaddr *)((caddr_t)ext + sizeof(*addr)));

	return;
}

void
kdebug_sadb_key(ext)
	struct sadb_ext *ext;
{
	struct sadb_key *key = (struct sadb_key *)ext;

	/* sanity check */
	if (ext == 0)
		printf("kdebug_sadb_key: NULL pointer was passed.\n");

	printf("sadb_key{ bits=%u reserved=%u }\n",
	    key->sadb_key_bits, key->sadb_key_reserved);
	printf("  key=");
#if 1
	kdebug_pdump((caddr_t)key + sizeof(struct sadb_key),
	             key->sadb_key_bits / 8,
	             YDUMP_HEX);
#else
	kdebug_pdump((caddr_t)key + sizeof(struct sadb_key),
	             key->sadb_key_len - sizeof(struct sadb_key),
	             YDUMP_HEX);
#endif

	return;
}

void
kdebug_sadb_policy(ext)
	struct sadb_ext *ext;
{
	struct sadb_x_policy *pl = (struct sadb_x_policy *)ext;

	/* sanity check */
	if (ext == 0)
		printf("kdebug_sadb_x_policy: NULL pointer was passed.\n");

	printf("sadb_x_policy{ type=%u", pl->sadb_x_policy_type);
	if (pl->sadb_x_policy_type == SADB_X_PL_IPSEC)
		printf(" esp/t=%u ah/t=%u esp/n=%u ah/n=%u }\n",
		    pl->sadb_x_policy_esp_trans,
		    pl->sadb_x_policy_ah_trans,
		    pl->sadb_x_policy_esp_network,
		    pl->sadb_x_policy_ah_network);
	else
		printf(" }\n");

	return;
}

void
kdebug_sockaddr(addr)
	struct sockaddr *addr;
{
	caddr_t ad;

	/* sanity check */
	if (addr == 0)
		printf("kdebug_sockaddr: NULL pointer was passed.\n");

	switch (addr->sa_family) {
	case PF_INET:
	{
		struct sockaddr_in *in = (struct sockaddr_in *)addr;
		printf("sockaddr{ len=%u family=%u port=%u }\n",
		    in->sin_len, in->sin_family, in->sin_port);
		ad = (caddr_t)&in->sin_addr;
	}
		break;
	case PF_INET6:
	{
		struct sockaddr_in6 *in6 = (struct sockaddr_in6 *)addr;
		ad = (caddr_t)&in6->sin6_addr;
	}
		break;
	default:
		return;
	}

	kdebug_pdump(ad, _ALENBYAF(addr->sa_family), YDUMP_HEX);

	return;
}

void
kdebug_vchar(v)
	vchar_t *v;
{
	/* sanity check */
	if (v == 0)
		printf("kdebug_vchar: NULL pointer was passed.\n");

	printf("vchar_t{ len=%u }\n", v->l);

	printf("  buf=");
	kdebug_pdump(v->v, v->l, YDUMP_HEX);

	return;
}

void
kdebug_pdump(buf, len, mode)
	caddr_t buf;
	int len;
	int mode;
{
	int i;

	if (mode == YDUMP_BIN) {
		for (i = 0; i < len; i++)
			printf("%c", (unsigned char)buf[i]);
		printf("\n");
	} else { /* HEX */
		for (i = 0; i < len; i++) {
			if (i != 0 && i % 32 == 0) printf("\n");
			if (i % 4 == 0) printf(" ");
			printf("%02x", (unsigned char)buf[i]);
		}
		if (i % 32 != 0) printf("\n");
	}

}

#endif /* defined(YIPS_DEBUG) */
