;;; mew-summary3.el --- Summary mode for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Oct  2, 1996

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Subfunctions
;;;

(defmacro mew-summary-prepare-draft (&rest body)
  "Common procedure to prepare a draft."
  `(progn
     (unwind-protect
	 (let ((find-file-hooks nil)
	       (inhibit-quit t))
	   ,@body
	   ;; XEmacs does not draw attachments unless sit for 0...
	   (mew-redraw)
	   ;; XEmacs does not draw toolbar, so...
	   (when (and mew-xemacs-p mew-icon-p
		      (specifier-instance default-toolbar-visible-p))
	     (set-specifier default-toolbar-visible-p nil)
	     (set-specifier default-toolbar-visible-p t)))
       (save-buffer)) ;; to make sure not to use this draft again
     (mew-set-file-modes (buffer-file-name))
     (mew-touch-folder mew-draft-folder)
     (message "Draft is prepared")))

(defsubst mew-summary-prepare-three-windows ()
  "Prepare three windows: Summary mode, Message mode, and Draft mode"
  (unless mew-use-other-frame-for-draft
    (if (get-buffer (mew-buffer-message))
	(delete-windows-on (mew-buffer-message)))
    (if (or mew-use-full-window (< (window-height) 25)) (delete-other-windows))
    (let ((split-window-keep-point t))
      (split-window-vertically))))

(defun mew-draft-multi-copy (draft files)
  (let* ((attach (mew-draft-to-attach draft))
	 (attachdir (mew-expand-folder attach)))
    (mew-check-directory attachdir)
    (while files
      (if mew-use-symbolic-link-for-forwarding
	  (mew-symbolic-link (car files) (mew-folder-new-message attach))
	(copy-file (car files) (mew-folder-new-message attach)))
      (setq files (cdr files)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Sending
;;;

(defun mew-summary-send (&optional to cc subject)
  "Write a message. A new draft is prepared in Draft mode."
  (interactive)
  (let ((draft (mew-folder-new-message mew-draft-folder)))
    (mew-current-set-window-config)
    (mew-window-configure 'draft)
    (mew-summary-prepare-draft
     (mew-draft-find-and-switch draft)
     (mew-delete-directory-recursively (mew-attachdir draft))
     (mew-draft-header subject nil to cc)
     (mew-draft-mode)
     (run-hooks 'mew-draft-mode-newdraft-hook))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Replying
;;;

(defun mew-subject-simplify (str &optional action-list no-replace)
  "A function to simplify a value of Subject: according to
'mew-subject-simplify-replace-alist'."
  (let ((case-fold-search t)
       regexp replace)
    (unless action-list (setq action-list mew-subject-simplify-replace-alist))
    (while action-list
      (setq regexp  (car (car action-list))
	    replace (if no-replace nil (cdr (car action-list)))
	    action-list (cdr action-list))
      (if (string-match regexp str)
	  (setq str (replace-match (if replace (eval replace) "") nil t str))))
    str))

(defun mew-summary-reply (&optional onlytofrom)
  "Answer to this message. A new draft is prepared in Draft mode. 
Mew automatically decides To: and Cc:. Addresses on To: and Cc:
are decided as follows:

    If From: of the message to be replied is not from me:
        Reply-To: does not exist in the message to be replied
            Copy From: of the message to be replied to To: (1)
            Copy To: and Cc: of the message to be replied to Cc: (2)
        Reply-To: exists in the message to be replied
            Copy From: and Reply-To: of the message to be replied to To: (3)
            Copy To: and Cc: of the message to be replied to Cc: (4)
    If From: of a message to be replied is from me:
        Copy To: of the message to be replied to To: (5)
        Copy Cc: of the message to be replied to Cc: (6)

You can customize which fields are copied in the case (1)-(6) with the
following variables:

    (1) mew-noreplyto-to-list
    (2) mew-noreplyto-cc-list
    (3) mew-replyto-to-list
    (4) mew-replyto-cc-list
    (5) mew-fromme-to-list
    (6) mew-fromme-cc-list

If executed with '\\[universal-argument]', only From: of the message is copied to To:.
"
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (mew-summary-toggle-disp-msg 'on)
    (mew-current-set-window-config)
    (let ((owin (selected-window))
	  (fld (mew-summary-folder-name))
	  (msg (mew-summary-message-number2))
	  cwin cbuf draft case
	  from reply-to to cc newsgroups subject in-reply-to references
	   encrypted fromme)
      (if (string= (mew-summary-folder-name) mew-draft-folder)
	  (message "Cannot reply to draft message")
	(setq draft (mew-folder-new-message mew-draft-folder))
	(mew-summary-prepare-draft
	 (mew-summary-prepare-three-windows)
	 (mew-draft-find-and-switch draft t)
	 (mew-delete-directory-recursively (mew-attachdir draft))
	 (setq cwin (selected-window)) ;; draft
	 (setq cbuf (window-buffer cwin))
	 (select-window owin)
	 ;; need to make a cache or a message buffer.
	 (let ((mew-use-full-window nil))
	   (mew-summary-display nil))
	 ;; see also mew-draft-cite
	 (set-buffer (or (save-excursion
			   (set-buffer (mew-buffer-message))
			   (if (mew-header-p) (current-buffer)))
			 ;; header exists only in cache if multipart
			 (mew-cache-hit fld msg)))
	 (when mew-case-guess-when-replied
	   (setq case (mew-draft-get-case-by-guess
		       mew-case-guess-when-replied-alist)))
	 (setq encrypted (mew-syntax-encrypted-p mew-decode-syntax))
	 (save-restriction
	   ;; if body contains ^L, header is not accessible.
	   ;; mew-header-* cannot widen essentially. So widen here.
	   (widen)
	   ;; now cache buffer
	   (setq from (mew-header-parse-address mew-from:))
	   (setq reply-to (mew-header-parse-address mew-reply-to:))
	   (cond 
	    (onlytofrom (setq to from))
	    ((mew-is-my-address mew-regex-my-address-list from)
	     ;; This message was sent by me. So, maintain To: and Cc:.
	     (setq fromme t)
	     (setq to (mew-header-parse-address-list2 mew-fromme-to-list))
	     (setq cc (mew-header-parse-address-list2 mew-fromme-cc-list))
	     (unless to (setq to (or reply-to from)))) ;; do not use list
	    (t
	     (cond 
	      (reply-to
	       (setq to (mew-header-parse-address-list2 mew-replyto-to-list))
	       (setq cc (mew-header-parse-address-list2 mew-replyto-cc-list)))
	      (t
	       (setq to (mew-header-parse-address-list2 mew-noreplyto-to-list))
	       (setq cc (mew-header-parse-address-list2 mew-noreplyto-cc-list))))))
	   (setq newsgroups (or (mew-header-get-value mew-followup-to:)
				(mew-header-get-value mew-newsgroups:)))
	   (if (and newsgroups (mew-case-equal newsgroups "poster"))
	       (setq newsgroups nil))
	   (setq subject (mew-header-get-value mew-subj:))
	   (if subject
	       (setq subject (mew-subject-simplify (concat mew-reply-string subject))))
	   (let ((old-message-id  (mew-header-get-value mew-message-id:))
		 (old-in-reply-to (mew-header-get-value mew-in-reply-to:))
		 (old-references  (mew-header-get-value mew-references:))
		 (regex "<[^>]+>")
		 (start 0) tmp-ref skip)
	     (if (and old-message-id (string-match regex old-message-id))
		 (setq old-message-id (match-string 0 old-message-id))
	       (setq old-message-id nil))
	     ;; Assuming that In-Reply-To: contains one ID.
	     (if (and old-in-reply-to (string-match regex old-in-reply-to))
		 (setq old-in-reply-to (match-string 0 old-in-reply-to))
	       (setq old-in-reply-to nil))
	     (if (null old-message-id)
		 () ;; we do not care even if old-references exist.
	       (setq in-reply-to old-message-id)
	       (if (null old-references)
		   (setq tmp-ref (if old-in-reply-to 
				     (list old-in-reply-to old-message-id)
				   (list old-message-id)))
		 (while (string-match "<[^>]+>" old-references start)
		   (setq start (match-end 0))
		   (setq tmp-ref (cons (match-string 0 old-references) tmp-ref)))
		 ;; described in old drums but not in RFC2822
		 (mew-addq tmp-ref old-in-reply-to)
		 (setq tmp-ref (nreverse (cons old-message-id tmp-ref))))
	       (if (integerp mew-references-max-count)
		   (setq skip (- (length tmp-ref) mew-references-max-count)))
	       (if (and (numberp skip) (> skip 0))
		   (setq tmp-ref (nthcdr skip tmp-ref)))
	       (setq references (mew-join "\n\t" tmp-ref)))))
	 ;;
	 (if (window-live-p cwin)
	     (select-window cwin) ;; draft
	   (pop-to-buffer cbuf))
	 (when case
	   (if mew-case-guess-addition
	       (setq case (mew-draft-add-case (mew-tinfo-get-case) case)))
	   (mew-tinfo-set-case case))
	 (mew-draft-header subject nil to cc newsgroups in-reply-to references
			   nil fromme)
	 (when (eq mew-summary-reply-position 'body)
	   (goto-char (mew-header-end))
	   (forward-line))
	 (mew-draft-mode encrypted)
	 (run-hooks 'mew-draft-mode-newdraft-hook)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Replaying with citation
;;;

(defun mew-summary-reply-with-citation (&optional onlytofrom)
  "Answer to this message. A new draft is prepared in Draft mode. 
And this message is automatically cited. See also 'mew-summary-reply'."
  (interactive "P")
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (let ((mew-summary-reply-position nil))
      (mew-summary-reply onlytofrom))
    ;; mew-draft-mode-hook may insert text.
    (save-excursion
      (goto-char (point-max))
      (run-hooks 'mew-before-cite-hook)
      (mew-draft-cite))
    ;; the cursor is after To:
    (cond
     ((eq mew-summary-reply-with-citation-position 'body)
      (goto-char (mew-header-end))
      (forward-line))
     ((eq mew-summary-reply-with-citation-position 'end)
      (goto-char (point-max)))))))
  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Forwarding
;;;

(defun mew-summary-forward ()
  "Forward this message to a third person. A new draft is prepared in 
Draft mode and this message is automatically attached."
  (interactive)
  (mew-summary-msg-or-part
   (mew-summary-not-in-draft
    (mew-current-set-window-config)
    (let* ((owin (selected-window))
	   (fld (mew-summary-folder-name))
	   (msg (mew-summary-message-number2))
	   (file (mew-expand-folder fld msg))
	   (draft (mew-folder-new-message mew-draft-folder))
	   (draftdir (file-name-nondirectory draft))
	   subject fwsubject cwin)
      (mew-summary-prepare-draft
       (mew-summary-prepare-three-windows)
       (mew-draft-find-and-switch draft t)
       (mew-delete-directory-recursively (mew-attachdir draft))
       (setq cwin (selected-window)) ;; draft
       (select-window owin)
       ;; need to make a cache or a message buffer.
       (let ((mew-use-full-window nil))
	 (mew-summary-display 'redisplay))
       ;;
       (set-buffer (or (save-excursion
			 (set-buffer (mew-buffer-message))
			 (if (mew-header-p) (current-buffer)))
		       ;; header exists only in cache if multipart
		       (mew-cache-hit fld msg)))
       (setq subject (mew-header-get-value mew-subj:))
       (if subject
	   (setq fwsubject (mew-subject-simplify (concat mew-forward-string subject))))
       (select-window cwin) ;; draft
       ;;
       (mew-draft-header fwsubject 'nl)
       (mew-draft-mode)
       (run-hooks 'mew-draft-mode-newdraft-hook)
       (mew-draft-multi-copy draft (list file))
       (setq mew-encode-syntax (mew-encode-syntax-initial-multi draftdir 1))
       (save-excursion
	 (mew-draft-prepare-attachments t)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Multi forwarding
;;;

(defun mew-summary-multi-forward ()
  "Forward messages marked with '@' to a third person. A new draft 
is prepared in Draft mode and this message is automatically 
attached."
  (interactive)
  (mew-summary-multi-msgs
   (mew-summary-not-in-draft
    (mew-current-set-window-config)
    (let* ((draft (mew-folder-new-message mew-draft-folder))
	   (draftdir (file-name-nondirectory draft)))
      (mew-summary-prepare-draft
       (mew-summary-prepare-three-windows)
       (mew-draft-find-and-switch draft t)
       (mew-delete-directory-recursively (mew-attachdir draft))
       (mew-draft-header nil 'nl)
       (mew-draft-mode)
       (run-hooks 'mew-draft-mode-newdraft-hook)
       (mew-draft-multi-copy draft FILES)
       (setq mew-encode-syntax
	     (mew-encode-syntax-initial-multi draftdir (length FILES)))
       (save-excursion
	 (mew-draft-prepare-attachments t)))))))

(provide 'mew-summary3)

;;; Copyright Notice:

;; Copyright (C) 1996-2002 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-summary3.el ends here
