/***************************************
  Meadow Font Manager
        Written by himi
***************************************/
#include "config.h"
#include "lisp.h"
#include "mw32term.h"
#include "mw32font.h"
#include "dispextern.h"
#include "frame.h"
#include "coding.h"
#include <commdlg.h>

extern Lisp_Object Qwidth, Qheight;

Lisp_Object Qbase, Qweight, Qfixed, Qitalic;
Lisp_Object Qfamily, Qraster, Qscalable;

int font_num = 0;
Lisp_Object Vw32_system_coding_system;
Lisp_Object Vw32_default_font_alist;

Lisp_Object Qw32_logfont, Qbdf_font;

static fs_fonts **w32_font_table = 0;

struct font_info *font_info_table;
int font_info_num = 0;
int font_info_size = 0;

static const unsigned char *default_font = "default";

#define DEFAULT_FONT_PROPERTY_MASK 0x4

static LOGFONT default_font_property[] =
{
  /*   Normal Font */
  {
    0, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, SHIFTJIS_CHARSET,
    OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
    FF_DONTCARE | FIXED_PITCH, "FixedSys"
  }
  ,
  /*   Bold Font */
  {
    0, 0, 0, 0, FW_BOLD, FALSE, FALSE, FALSE, SHIFTJIS_CHARSET,
    OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
    FF_DONTCARE | FIXED_PITCH, "FixedSys"
  }
  ,
  /*   Italic Font */
  {
    0, 0, 0, 0, FW_NORMAL, TRUE, FALSE, FALSE, SHIFTJIS_CHARSET,
    OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
    FF_DONTCARE | FIXED_PITCH, "FixedSys"
  }
  ,
  /*   Italic-Bold Font */
  {
    0, 0, 0, 0, FW_BOLD, TRUE, FALSE, FALSE, SHIFTJIS_CHARSET,
    OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
    FF_DONTCARE | FIXED_PITCH, "FixedSys"
  }
};


extern void w32_set_font (struct frame *f, Lisp_Object arg, Lisp_Object oldval);
extern int check_w32gui ();

int
get_valid_font_property(fs_fonts *sf, int property)
{
  LOGFONT *lf;
  int prop = property;
  int mask = FONT_PROPERTY_MASK;

  do {
    lf = MWF_GET_LOGFONT(sf, prop);
    mask >>= 1;
    prop &= mask;
  } while(!lf && mask);

  return prop;
}

LPLOGFONT
get_logfont(fs_fonts *sf, int property)
{
  LOGFONT *lf;
  int prop;

  prop = get_valid_font_property(sf, property);

  lf = MWF_GET_LOGFONT(sf, prop);

  if (!lf)
    {
      prop = property & DEFAULT_FONT_PROPERTY_MASK;
      lf = &default_font_property[prop];
      lf->lfWidth = sf->hmetric;
      lf->lfHeight = sf->vmetric;
    }
  return lf;
}

HFONT
get_hfont(fs_fonts *sf, int property)
{
  LOGFONT *lf;
  HFONT hf;

  property = get_valid_font_property(sf, property);
  if (hf = MWF_GET_FONT(sf, property))
    return hf;
  lf = get_logfont(sf, property);

  hf = CreateFontIndirect(lf);
  MWF_GET_FONT(sf, property) = hf;
  return hf;
}

fs_fonts
*mw32_get_font(const unsigned char *name)
{
  int i;
  fs_fonts **sf;

  sf = w32_font_table;
  for (i = 0;i < MWF_NUM; i++, sf++)
    {
      if (strcmp((*sf)->name, name) == 0)
	{
	  return *sf;
	}
    }
  return 0;
}

int
set_ccl_method(ccl_id_type cclid, struct ccl_program *encoder)
{
  extern Lisp_Object Vccl_program_table;
  extern Lisp_Object Qccl_program_idx;
  Lisp_Object encoding, ccl_id, ccl_prog;

  if (!cclid.data) return 0;
  encoding = Fintern_soft(make_string(cclid.data, cclid.size), Qnil);

#if 0
  if (Fcoding_system_p(encoding))
    {
      struct coding_system* lpmccode;

      lpmccode = (struct coding_system*) xmalloc(sizeof(struct coding_system));

      setup_coding_system(encoding, lpmccode);
      if ((lpmccode->type) == coding_type_ccl)
	{
	  xfree(lpmccode);
	  return 0;
	}
      encoder->size = 0;
      encoder->prog = (Lisp_Object *)lpmccode;
    }
  else
    {
      setup_ccl_program(encoder, encoding);
    }
#else
  if (SYMBOLP(encoding) &&
      (SYMBOLP (encoding = Fget (encoding, Qccl_program))) &&
      (setup_ccl_program(encoder, encoding) >= 0))
    return 1;
#endif

  return 0;
}

void
set_font_info(fs_fonts *sf, struct font_info *fontp, int index_num)
{
  int size;

  fontp->font = sf;
  size = strlen(sf->name) + 1;
  if (index_num == -1)
    {
      xfree(fontp->name);
      xfree(fontp->full_name);
      if ((sf->ccl_method.data) && (fontp->font_encoder))
	{
	  /* Caution!!!! */
	  if (!set_ccl_method(sf->ccl_method, fontp->font_encoder))
	    {
	      xfree(fontp->font_encoder);
	      fontp->font_encoder = NULL;
	    }
	}
    }
  else
    {
      fontp->font_idx = index_num;
    }
  fontp->name = (char *) xmalloc (size);
  fontp->full_name = (char *) xmalloc (size);
  memcpy (fontp->name, sf->name, size);
  memcpy (fontp->full_name, sf->name, size);
  fontp->encoding[1] = sf->encoding_type;

  fontp->size = sf->hmetric;
  fontp->height = sf->vmetric;
  fontp->baseline_offset = 0;
  fontp->relative_compose = sf->relative_compose;
  fontp->default_ascent = sf->default_ascent;
}  

struct font_info*
w32_load_font(struct frame *f,
	      unsigned char *name, int size)
{
  fs_fonts *sf;
  struct font_info *fontp;
  
  sf = mw32_get_font(name);
  if (!sf) return (struct font_info *) NULL;
  if (sf->fontip) return sf->fontip;

  if (font_info_size == 0)
    {
      font_info_size = 16;
      W32_FONT_INFO_TABLE = (struct font_info *)
	xmalloc(font_info_size * sizeof(struct font_info));
    }
  else if (font_info_size <= W32_FONT_INFO_NUM)
    {
      font_info_size *= 2;
      W32_FONT_INFO_TABLE = (struct font_info *)
	xrealloc(W32_FONT_INFO_TABLE, font_info_size * sizeof(struct font_info));
    }
  fontp = &W32_FONT_INFO_TABLE[W32_FONT_INFO_NUM];
  set_font_info(sf, fontp, W32_FONT_INFO_NUM);
  W32_FONT_INFO_NUM++;

  return fontp;
}

#if 0
void
w32_unload_font(struct frame *f, FONT_TYPE *font)
{
#if 0
  HFONT hf;
  hf = MWF_GET_FONT(font->fs_font, font->property_index);

  if (hf)
    {
      DeleteObject(*phf);
      *phf = NULL;
    }
#endif
  xfree(font);

  return;
}
#endif

void
expand_font_table(int num)
{
  int i;
  fs_fonts **pfs;
  fs_fonts *fs;
  
  w32_font_table =
    (fs_fonts **)xrealloc(w32_font_table,
			  (MWF_NUM + num) * sizeof(fs_fonts*));
  pfs = w32_font_table + MWF_NUM;
  fs = (fs_fonts *) xmalloc(num * sizeof(fs_fonts));
  MWF_NUM += num;

  for (i = 0;i < num;i++, fs++, pfs++)
    {
      *pfs = fs;
    }
}

void
w32_find_ccl_program (struct font_info *fontp)
{
  fs_fonts *sf;
  Lisp_Object encoding;

  sf = mw32_get_font(fontp->name);

  if (!sf) return;
  fontp->font_encoder
    = (struct ccl_program *) xmalloc (sizeof (struct ccl_program));
  if (!set_ccl_method(sf->ccl_method, fontp->font_encoder))
    {
      xfree(fontp->font_encoder);
      fontp->font_encoder = NULL;
    }
  return;
}

Lisp_Object
get_font_parameter(Lisp_Object param, Lisp_Object alist)
{
  Lisp_Object elem;

  elem = Fassq(param, alist);

  if (NILP(elem))
    {
      elem = Fassq(param, Vw32_default_font_alist);
      if (NILP(elem)) return Qnil;
    }

  return Fcdr(elem);
}

void 
internal_add_font(int location,
		  const unsigned char *name,
		  int width, int height, int base, int overhang, 
		  int encoding_type,
		  int relative_compose, int default_ascent,
		  ccl_id_type cclid)
{
  fs_fonts *fp;
  int i;
  fp = (w32_font_table)[location];
  fp->name = (unsigned char *)xmalloc(strlen(name) + 1);
  fp->vmetric = height;
  fp->hmetric = width;
  fp->font_base = base;
  fp->overhang = overhang;
  fp->encoding_type = encoding_type;
  fp->relative_compose = relative_compose;
  fp->default_ascent = default_ascent;
  fp->ccl_method = cclid;
  fp->fontip = NULL;
  strcpy(fp->name, name);
  for (i = 0;i < FONT_PROPERTY_NUM;i++)
    {
      fp->font_properties[i] = (HFONT) 0;
      fp->logfont_properties[i] = (LPLOGFONT) 0;
    }
}

Lisp_Object
logfont_to_lisp_object(LOGFONT *lf)
{
  Lisp_Object logfont[13];
  /* [0] is identifier.
     [1] is name.
     [2] is width. 
     [3] is height. 
     [4] is weight.
     [5] is orientation.
     [6] is italic.
     [7] is underline.
     [8] is strikeout.
     [9] is charset.
     [10] is quality. 
     [11] is OutPrecision 
     [12] is PitchAndFamily */

  unsigned char *buf;
  struct coding_system coding;
  int nbytes, size, bufsize;

  nbytes = strlen (lf->lfFaceName);
  setup_coding_system (Fcheck_coding_system(Vw32_system_coding_system), &coding);
  bufsize = decoding_buffer_size (&coding, nbytes); 
  buf = (unsigned char *)alloca (bufsize);
  decode_coding (&coding, lf->lfFaceName, buf, nbytes, bufsize);
  size = coding.produced;
  logfont[0] = Qw32_logfont;
  logfont[1] = make_string(buf, size);

  logfont[2] = make_number(lf->lfWidth);
  logfont[3] = make_number(lf->lfHeight);
  XSETFASTINT(logfont[4], lf->lfWeight);
  XSETFASTINT(logfont[5], lf->lfOrientation);
  logfont[6] = (lf->lfItalic) ? Qt:Qnil;
  logfont[7] = (lf->lfUnderline) ? Qt:Qnil;
  logfont[8] = (lf->lfStrikeOut) ? Qt:Qnil;
  XSETFASTINT(logfont[9], lf->lfCharSet);
  XSETFASTINT(logfont[10], lf->lfQuality);
  XSETFASTINT(logfont[11], lf->lfOutPrecision);
  XSETFASTINT(logfont[12], lf->lfPitchAndFamily);

  return Flist(XFASTINT(13), logfont);
}

int
encode_logfont_name(Lisp_Object src, char *dest)
{
  int bufsize, size;
  struct coding_system coding;
  char *buf;

  setup_coding_system (Fcheck_coding_system (Vw32_system_coding_system), &coding);
  bufsize = encoding_buffer_size (&coding, LISPY_STRING_BYTES(src) + 1);

  buf = alloca (bufsize);
  /* LOGFONT is NEVER accessed by lisp interpreter. */
  encode_coding (&coding, XSTRING(src)->data, buf,
		 LISPY_STRING_BYTES(src), bufsize);
  size = coding.produced;
  if (size >= LOGFONT_FACENAME_MAX)
    error("The name is too long. Max is %d", LOGFONT_FACENAME_MAX);

  buf[size] = '\0';
  if (dest)
    strcpy(dest, buf);

  return size;
}

LOGFONT
lisp_object_to_logfont(Lisp_Object font)
{
  LOGFONT lf;
  Lisp_Object tmpcdr, tmpcar;

  tmpcdr = Fcdr (font);
  tmpcar = Fcar (tmpcdr);

  encode_logfont_name(tmpcar, lf.lfFaceName);

  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfWidth = XINT(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfHeight = XINT(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfWeight = XFASTINT(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfOrientation = XFASTINT(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfItalic = !NILP(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfUnderline = !NILP(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfStrikeOut = !NILP(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfCharSet = XFASTINT(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfQuality = XFASTINT(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfOutPrecision = XFASTINT(tmpcar);
  tmpcdr = Fcdr (tmpcdr);
  tmpcar = Fcar (tmpcdr);
  lf.lfPitchAndFamily = XFASTINT(tmpcar);

/* Set the default value... */
  lf.lfEscapement = lf.lfOrientation;
  lf.lfClipPrecision = CLIP_DEFAULT_PRECIS;

  return lf;
}

ccl_id_type
get_ccl_id (Lisp_Object encode)
{
  ccl_id_type cclid;

  if (NILP(encode))
    {
      cclid.data = NULL;
    } 
  else
    {
      Lisp_Object symname;
    
      symname = Fsymbol_name(encode);
      cclid.size = LISPY_STRING_BYTES(symname);
      cclid.data = (char*)xmalloc(cclid.size);
      memcpy(cclid.data, XSTRING(symname)->data, cclid.size);
    }

  return cclid;
}

struct font_info*
w32_get_font_info (FRAME_PTR f, int font_idx)
{
  return &W32_FONT_INFO_TABLE[font_idx];
}

struct font_info*
w32_query_font (struct frame *f, char *fontname)
{
  int i;
  struct font_info *pfi;

  pfi = W32_FONT_INFO_TABLE;

  for (i = 0;i < W32_FONT_INFO_NUM;i++, pfi++)
    {
      if (strcmp(pfi->name, fontname) == 0) return pfi;
    }

  return NULL;
}

int
w32_score_logfont(logfont_candidates *plc, Lisp_Object request)
{
  int score;
  Lisp_Object key, val, tem;
  char *str;

  CHECK_LIST(request, 0);
  key = XCONS(request)->car;
  val = XCONS(request)->cdr;
  if (CONSP(key))
    {
      CHECK_NUMBER(val, 0);
      score = XINT(val);
      tem = key;
      key = XCONS(tem)->car;
      val = XCONS(tem)->cdr;
    }
  else
    score = 1;

  if (EQ(key, Qwidth))
    {
      CHECK_NUMBER(val, 0);
      if ((plc->type == 0)
	  && ((XINT(val) != plc->width)))
	score = 0;
    }
  else if (EQ(key, Qheight))
    {
      CHECK_NUMBER(val, 0);
      if ((plc->type == 0)
	  && ((XINT(val) != plc->height)))
	score = 0;
    }
  else if (EQ(key, Qweight))
    {
      CHECK_NUMBER(val, 0);
      if ((plc->type == 0)
	  && ((XINT(val) != plc->lf.lfWeight)))
	score = 0;
    }
  else if (EQ(key, Qbase))
    {
      CHECK_NUMBER(val, 0);
      if (XINT(val) != plc->base)
	score = 0;
    }
  else if (EQ(key, Qitalic))
    {
      if ((NILP(val) && (plc->lf.lfItalic))
	  || (!NILP(val) && !(plc->lf.lfItalic)))
	score = 0;
    }
  else if (EQ(key, Qfixed))
    {
      int fixedp = (plc->lf.lfPitchAndFamily & FIXED_PITCH);
      if ((NILP(val) && fixedp)
	  || (!NILP(val) && !fixedp))
	score = 0;
    }
  else if (EQ(key, Qfamily))
    {
      if (NILP(Fstring_equal(val, plc->family)))
	score = 0;
    }

  return score;
}
  
DEFUN("w32-check-logfont", Fw32_check_logfont, Sw32_check_logfont, 1, 1, 0,
      "Check validity of LOGFONT. \n\
A correct LOGFONT must be a list that have 2 or 12 elements.\n\
(bdf-font path) or \n\
(w32-logfont name width height weight orientation italic underline\n\
strikeout charset quality OutPrecision PitchAndFamily)")
     (logfont)
     Lisp_Object logfont;
/* The w32-logfont is to specify type of logfont.
The width must be a string that describes the name of its font.\n\
The height must be a number that describes the width of its font.\n\
The weight must be a number that describes the weight.(1-1000)\n\
The orientation must be a number that describes the orientation.(angle)\n\
If the italic is nil, it is not italic font.\n\
If the strikeout is nil, it doesn't have a strikeout line.\n\
The charset must be a number that describes the charset that it uses.\n\
(128 is SJIS, 0 is ANSI)\n\
The quality must be a number that describes its quality.(0-2) */
{
  Lisp_Object tmpcar, tmpcdr, type;

  CHECK_LIST(logfont, 0);
  tmpcar = Fcar(logfont);
  tmpcdr = Fcdr(logfont);
  CHECK_SYMBOL(tmpcar, 1); /* w32-logfont or bdf-font */
  type = tmpcar;
  tmpcar = Fcar(tmpcdr);
  tmpcdr = Fcdr(tmpcdr);
  CHECK_STRING(tmpcar, 1); /* name  or path */

  if (type == Qw32_logfont)
    {
      encode_logfont_name(tmpcar, NULL);
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar,2); /* width */
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar,3); /* height */
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar, 4); /* weight */
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar, 5); /* orientation */
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      if (!((EQ(tmpcar, Qnil)) || (EQ(tmpcar, Qt)))) /* italic */
	{
	  error("italic cell must be nil or t");
	}
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      if (!((EQ(tmpcar, Qnil)) || (EQ(tmpcar, Qt)))) /* underline */
	{
	  error("underline cell must be nil or t");
	}
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      if (!((EQ(tmpcar, Qnil)) || (EQ(tmpcar, Qt)))) /* strikeout */
	{
	  error("strikeout cell must be nil or t");
	}
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar, 6); /* charset */
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar, 7); /* quality */
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar, 8); /* OutPrecision */
      tmpcar = Fcar(tmpcdr);
      tmpcdr = Fcdr(tmpcdr);
      CHECK_NUMBER(tmpcar, 9); /* PitchAndFamily */
    }
  else if (type != Qbdf_font)
    {
      error ("undefined type of logfont %s", XSYMBOL(type)->name);
    }

  if (!NILP(tmpcdr))
    {
      error ("Invalid logfont. Its length is too long.");
    }
  return Qt;
}


/* NOTE! Encoding Type -1 Problem!!!!! */
/* font manager */
DEFUN("w32-add-font", Fw32_add_font, Sw32_add_font, 2, 2, 0,
      "Add a font. NAME is a name of the font.\n\
ALIST is parameters to use this font.")
     (name, alist)
{
  Lisp_Object width, height, base, overhang, encoding_type;
  Lisp_Object relative_compose, default_ascent, encoder;

  CHECK_STRING(name, 0);
  CHECK_LIST(alist, 1);

  width = get_font_parameter(Qwidth, alist);
  height = get_font_parameter(Qheight, alist);
  base = get_font_parameter(Qbase, alist);
  overhang = get_font_parameter(intern("overhang"), alist);
  encoding_type = get_font_parameter(intern("encoding-type"), alist);

  if (NILP(width) || NILP(height) || NILP(base) ||
      NILP(overhang) || NILP(encoding_type))
    {
      error("width, height, base, overhang, encoding-type must be specified!");
    }

  relative_compose = get_font_parameter(intern("relative-compose"), alist);
  default_ascent = get_font_parameter(intern("default-ascent"), alist);
  encoder = get_font_parameter(intern("encoder"), alist);

  CHECK_NUMBER(width, 1);
  CHECK_NUMBER(height, 2);
  CHECK_NUMBER(base, 3);
  CHECK_NUMBER(overhang, 4);
  CHECK_NUMBER(encoding_type, 5);

  if (NILP(relative_compose)) relative_compose = XFASTINT(0);
  if (NILP(default_ascent)) default_ascent = XFASTINT(0);

  CHECK_NUMBER(relative_compose, 5);
  CHECK_NUMBER(default_ascent, 6);

  if (!NILP(encoder))
    {
      if (!ENCODING_METHOD_P(encoder))
	{
	  error("This encode is NOT valid!");
	}
    }

  if (mw32_get_font(XSTRING(name)->data))
    {
      error("Already registerd %s font.", XSTRING(name)->data);
    }

  expand_font_table(1);
  internal_add_font(MWF_NUM - 1,
		    XSTRING(name)->data,
		    XFASTINT(width),
		    XFASTINT(height),
		    XFASTINT(base),
		    XFASTINT(overhang),
		    XINT(encoding_type),
		    XINT(relative_compose),
		    XFASTINT(default_ascent),
		    get_ccl_id(encoder));


  return Qnil;
}

DEFUN("w32-change-font-attribute", Fw32_change_font_attribute,
      Sw32_change_font_attribute, 2, 2, 0,
      "Change the font attribute.")
     (name, alist)
     Lisp_Object name, alist;
{
  fs_fonts *fontp;
  Lisp_Object width, height, base, overhang, encoding_type;
  Lisp_Object relative_compose, default_ascent, encoder;
  CHECK_STRING(name, 0);
  CHECK_LIST(alist, 1);

  width = get_font_parameter(Qwidth, alist);
  height = get_font_parameter(Qheight, alist);
  base = get_font_parameter(Qbase, alist);
  overhang = get_font_parameter(intern("overhang"), alist);
  encoding_type = get_font_parameter(intern("encoding-type"), alist);

  if (NILP(width) || NILP(height) || NILP(base) ||
      NILP(overhang) || NILP(encoding_type))
    {
      error("width, height, base, overhang, encoding-type must be specified!");
    }

  relative_compose = get_font_parameter(intern("relative-compose"), alist);
  default_ascent = get_font_parameter(intern("default-ascent"), alist);
  encoder = get_font_parameter(intern("encoder"), alist);

  CHECK_STRING(name, 0);
  CHECK_NUMBER(width, 1);
  CHECK_NUMBER(height, 2);
  CHECK_NUMBER(overhang, 3);
  CHECK_NUMBER(base, 4);
  CHECK_NUMBER(encoding_type, 5);

  if (NILP(relative_compose)) relative_compose = XFASTINT(0);
  if (NILP(default_ascent)) default_ascent = XFASTINT(0);
  CHECK_NUMBER(relative_compose, 6);
  CHECK_NUMBER(default_ascent, 7);

  if (!NILP(encoder))
    {
      if (!ENCODING_METHOD_P(encoder))
	{
	  error("This encode is NOT valid!");
	}
    }

  fontp = mw32_get_font(XSTRING(name)->data);

  fontp->hmetric = XFASTINT(width);
  fontp->vmetric = XFASTINT(height);
  fontp->overhang = XFASTINT(overhang);
  fontp->font_base = XFASTINT(base);
  fontp->encoding_type = XINT(encoding_type);
  fontp->relative_compose = XINT(relative_compose);
  fontp->default_ascent = XFASTINT(default_ascent);
  fontp->ccl_method = get_ccl_id(encoder);

  if (fontp->fontip) set_font_info(fontp, fontp->fontip, -1);

  return Qnil;
}

DEFUN("w32-font-list", Fw32_font_list, Sw32_font_list, 0, 0, 0,
      "Return a list of all w32 font. ")
     ()
{
  int i;
  Lisp_Object val = Qnil;

  for(i = 0;i < MWF_NUM;i++)
    {
      val = Fcons(build_string((w32_font_table)[i]->name), val);
    }
  return val;
}
  
/* font property manager */
DEFUN("w32-change-font-logfont", Fw32_change_font_logfont,
      Sw32_change_font_logfont, 3, 3, 0,
      "Change font property. \n\
The first arg is a name of the font where the property is changed.\n\
The second ARG is a number of the property.\n\
The third ARG is the logfont infomation.")
     (font, number, logfont)
{
  fs_fonts *fontp;
  LPLOGFONT plf, pnlf;
  HFONT hf;
  int i, prop_num;

  CHECK_STRING(font, 0);
  CHECK_NUMBER(number, 1);

  prop_num = XFASTINT(number);
  if (prop_num >= FONT_PROPERTY_NUM)
    {
      error("Too large font property number.");
    }

  fontp = mw32_get_font(XSTRING(font)->data);

  if (!fontp)
    {
      error ("Such font does not exist!");
    }

  plf = MWF_GET_LOGFONT(fontp, XFASTINT(number));
  hf = MWF_GET_FONT(fontp, XFASTINT(number));
  Fw32_check_logfont(logfont);

  if (EQ(XCONS(logfont)->car, Qbdf_font))
    {
      char *data;
      data = XSTRING(XCONS(XCONS(logfont)->cdr)->car)->data;
      pnlf = (LPLOGFONT) mw32_init_bdf_font(data);
      if (!pnlf)
	error("Can't open font. %s", data);
      MWF_GET_LOGFONT(fontp, XFASTINT(number)) = pnlf;
      MWF_GET_FONT(fontp, XFASTINT(number)) = INVALID_HANDLE_VALUE;
    }
  else
    {
      if (!plf)
	{
	  pnlf = (LPLOGFONT)xmalloc(sizeof(LOGFONT));
	  MWF_GET_LOGFONT(fontp, XFASTINT(number)) = pnlf;
	}
      else
	pnlf = plf;

      *pnlf = lisp_object_to_logfont(logfont);
      MWF_GET_FONT(fontp, XFASTINT(number)) = (HANDLE) NULL;

      return Qnil;
    }

  /* delete old font handle */
  if (!hf) return Qnil;
  if (hf == INVALID_HANDLE_VALUE)
    mw32_free_bdf_font((bdffont*) plf);
  else
    DeleteObject(hf);

  return Qnil;
}

DEFUN ("w32-get-font-logfont", Fw32_get_font_logfont, 
       Sw32_get_font_logfont, 2, 2, 0,
       "Get the logfont of the font you specified.\n\
The first arg is a name of the font.\n\
The second arg is a number of the property.\n\
If this property has no logfont, return nil")
(font, number)
{
  fs_fonts *fontp;
  int i, prop_num;
  LOGFONT *lf;

  CHECK_STRING(font, 0);
  CHECK_NUMBER(number, 1);
  prop_num = XFASTINT(number);
  if (prop_num >= FONT_PROPERTY_NUM)
    {
      error("Too large font property number.");
    }

  fontp = mw32_get_font(XSTRING(font)->data);
  if (fontp)
    {
      lf = MWF_GET_LOGFONT(fontp, XFASTINT(number));
      if (MWF_GET_FONT(fontp, XFASTINT(number)) == INVALID_HANDLE_VALUE)
	{
	  if  (lf)
	    return build_string(BDF_FONT_FILE(lf));
	}
      else if (lf)
	{
	  return logfont_to_lisp_object(lf);
	}
      return Qnil;
    }
  error ("Such font does not exist!");

  return Qnil;
}

DEFUN ("w32-get-font-info", Fw32_get_font_info,
       Sw32_get_font_info, 1, 1, 0,
       "Get the font infomation you specified.\n\
NAME is a name of the font.")
     (name)
{
  fs_fonts *fontp;
  Lisp_Object ret = Qnil;

  CHECK_STRING(name, 0);

  fontp = mw32_get_font(XSTRING(name)->data);
  if (fontp)
    {
      Lisp_Object encoding;

      if (fontp->ccl_method.data)
	{
	  encoding = Fintern_soft(make_string(fontp->ccl_method.data,
					      fontp->ccl_method.size),
				  Qnil);
	  if (!ENCODING_METHOD_P(encoding))
	    encoding = Qnil;
	}
      else
	encoding = Qnil;

      store_in_alist(&ret, Qwidth,
		     make_number(fontp->hmetric));
      store_in_alist(&ret, Qheight,
		     make_number(fontp->vmetric));
      store_in_alist(&ret, Qbase,
		     make_number(fontp->font_base));
      store_in_alist(&ret, intern("overhang"),
		     make_number(fontp->overhang));
      store_in_alist(&ret, intern("encoding-type"),
		     make_number(fontp->encoding_type));
      store_in_alist(&ret, intern("relative-compose"),
		     make_number(fontp->relative_compose));
      store_in_alist(&ret, intern("default-ascent"),
		     make_number(fontp->default_ascent));
      store_in_alist(&ret, intern("encoder"), encoding);

      return ret;
    }
  error ("Such font does not exist!");

  return Qnil;
}

DEFUN ("w32-get-logfont-info", Fw32_get_logfont_info, 
       Sw32_get_logfont_info, 1, 2, 0,
       "Get the metric of the logfont you specified.\n\
Return an a-list that consists of width, height, base, overhang,\n\
charset-num, and max-width elements.")
     (logfont, frame)
     Lisp_Object logfont, frame;
{
  LOGFONT lf;
  HFONT hf;
  Lisp_Object ret = Qnil;

  Fw32_check_logfont(logfont);

  if (EQ(XCONS(logfont)->car, Qbdf_font))
    {
      return mw32_get_bdf_font_info (XCONS(XCONS(logfont)->cdr)->car);
    }
  else
    {

      lf = lisp_object_to_logfont(logfont);
      
      if (hf = CreateFontIndirect(&lf))
	{
	  HDC hdc;
	  HANDLE oldobj;
	  TEXTMETRIC tm;
	  HWND hwnd;
	  int flag;

	  if (FRAMEP(frame))
	    {
	      hwnd = FRAME_W32_WINDOW(XFRAME(frame));
	    }
	  else 
	    hwnd = ROOT_WINDOW;

	  hdc = MyGetDC(hwnd);
	  oldobj = SelectObject(hdc, hf);
	  flag = GetTextMetrics(hdc, &tm);
	  SelectObject(hdc, oldobj);
	  DeleteObject(hf);
	  ReleaseDC(hwnd, hdc);
	  if (!flag)
	    error("Fail to get metrics!!, FaceName:%s",
		  lf.lfFaceName);

	  store_in_alist(&ret, Qwidth,
			 make_number((int)tm.tmAveCharWidth));
	  store_in_alist(&ret, Qheight,
			 make_number((int)tm.tmHeight));
	  store_in_alist(&ret, Qbase,
			 make_number((int)tm.tmAscent));
	  store_in_alist(&ret, intern("overhang"),
			 make_number((int)tm.tmOverhang));
	  store_in_alist(&ret, intern("charset-num"),
			 make_number((int)tm.tmCharSet));
	  store_in_alist(&ret, intern("max-width"),
			 make_number((int)tm.tmMaxCharWidth));
	  return ret;
	}
    }
  error("unable to create font!");

  return Qnil;
}

int CALLBACK mw32_enumfontfamilyproc(ENUMLOGFONT *lpelf,
				     NEWTEXTMETRIC *lpntm,
				     int fonttype,
				     Lisp_Object *result)
{
  Lisp_Object rlist[4];
  Lisp_Object rf;

  int bufsize, size;
  int fullname_size, style_size;
  unsigned char *buf;
  struct coding_system coding;

  fullname_size = strlen (lpelf->elfFullName);
  style_size = strlen (lpelf->elfStyle);

  setup_coding_system (Fcheck_coding_system(Vw32_system_coding_system), &coding);
  bufsize = decoding_buffer_size (&coding, max (fullname_size, style_size));
  buf = (unsigned char *)alloca (bufsize);

  /* This procedure may cause GC, so we should make it first. */
  rlist[3] = logfont_to_lisp_object (&(lpelf->elfLogFont));

  decode_coding (&coding, lpelf->elfFullName, buf,
		 fullname_size, bufsize);
  size = coding.produced;
  rlist[0] = make_string (buf, size);

  decode_coding (&coding, lpelf->elfStyle, buf, 
		 style_size, bufsize);
  size = coding.produced;
  rlist[1] = make_string (buf, size);

  if (fonttype & RASTER_FONTTYPE)
    rlist[2] = Qraster;
  else
    rlist[2] = Qscalable;

  /* To protect data from GC, I connect these expressions. */
  /* result is protected from GC, so we can refer it by pointer. */
  *result = Fcons (Flist (XFASTINT (4), rlist), *result);

  return 1;

}

DEFUN ("w32-enum-logfont", Fw32_enum_logfont,
       Sw32_enum_logfont, 0, 2, 0,
       "Enumerate logfonts in the system.\n\
FAMILY is a family name of enumerated logfonts.\n\
If this argument is nil, enumerate all families.\n\
If DEVICE is nil, the common device context is\n\
used for enumeration.\n\
This function returns results, as follow.\n\
((FULLNAME STYLENAME FONTTYPE LOGFONT) \n\
 (FULLNAME STYLENAME FONTTYPE LOGFONT) \n\
 ....)")
(family, device)
     Lisp_Object family, device;
{
  struct gcpro gcpro1;
  int device_type;
  HWND hwnd;
  HDC hdc;
  Lisp_Object result = Qnil;

  if (NILP (device)) device_type = 0;
  else if (EQ (device, intern ("printer")))
    error ("Not yet supported device: %s", device);
  else if (EQ (device, intern ("display"))) device_type = 0;
  else
    error ("Unknown device: %s", device);

  if (device_type == 0)
    {
      hwnd = GetDesktopWindow ();
      hdc = MyGetDC(hwnd);
    }
  
  GCPRO1 (result);
  if (NILP (family))
    EnumFontFamilies (hdc, NULL,
		      (FONTENUMPROC) mw32_enumfontfamilyproc,
		      (LPARAM) &result);
  else
    {
      struct coding_system coding;
      int size, bufsize;
      unsigned char *buf;

      CHECK_STRING (family, 0);
      setup_coding_system (Fcheck_coding_system (Vw32_system_coding_system),
			   &coding);
      bufsize = encoding_buffer_size (&coding, LISPY_STRING_BYTES(family) + 1);
      buf = alloca (bufsize);
      encode_coding (&coding, XSTRING(family)->data, buf,
		     LISPY_STRING_BYTES(family), bufsize);
      size = coding.produced;
      buf[size] = '\0';
      EnumFontFamilies (hdc, buf,
			(FONTENUMPROC) mw32_enumfontfamilyproc,
			(LPARAM) &result);
    }

  UNGCPRO;
  if (device_type == 0)
    {
      ReleaseDC (hwnd, hdc);
    }

  return result;
}

DEFUN ("w32-query-get-logfont", Fw32_query_get_logfont,
       Sw32_query_get_logfont, 0, 0, 0,
       "Return the font that you selected with Font Select Dialog Box.")
     ()
{
  CHOOSEFONT cf;
  LOGFONT lf;
  unsigned char buf[100];
  int ret;

  cf.lStructSize = sizeof (CHOOSEFONT);
  cf.hwndOwner = FRAME_W32_WINDOW(selected_frame);
  cf.hDC = (HDC)NULL;
  cf.lpLogFont = &lf;
  cf.Flags = CF_SCREENFONTS | CF_FIXEDPITCHONLY | 
    CF_FORCEFONTEXIST | CF_EFFECTS;
  cf.rgbColors = RGB(0,0,0);
  cf.lpfnHook = (LPCFHOOKPROC)NULL;
  cf.lpTemplateName = (LPSTR)NULL;
  cf.hInstance = (HINSTANCE) NULL;
  cf.lpszStyle = (LPSTR)NULL;
  cf.nFontType = SCREEN_FONTTYPE;

  ShowCursor (TRUE);
  ret = ChooseFont (&cf);
  ShowCursor (FALSE);


  if (ret)
    return logfont_to_lisp_object (&lf);
  else
    return Qnil;
}

DEFUN ("w32-score-logfont-candidates", Fw32_score_logfont_candidates,
       Sw32_score_logfont_candidates, 2, 2, 0,
       "Score logfont candidates from requests.\n\
The first argument REQUESTS is a list that consists of either\n\
  ((KEY . VALUE) . POINT)\n\
or\n\
  (KEY . VALUE).\n\
In the latter form, its POINT is regarded as 1.\n\
\n\
The second argument CANDIDATES is a list that consists of\n\
  (FULLNAME STYLENAME FONTTYPE LOGFONT)\n\
type cell, and its format is equivalent to the result format of\n\
(w32-enum-logfont).\n\
\n\
On each element of CANDIDATES and each element of REQUESTS,\n\
this function add POINT to a score of the element of CANDIDATES\n\
if it conforms the elements of REQUEST.\n\
\n\
The meaningful KEY is one of the followings.\n\
   family (The VALUE is a string.)\n\
   width  (The VALUE is a number.)\n\
   height (The VALUE is a number.)\n\
   base   (The VALUE is a number.)\n\
   width  (The VALUE is a number.)\n\
   weight (The VALUE is a number.)\n\
   italic (If The VALUE is non-nil, italic fonts conform, vice-virsa.)\n\
   fixed  (If The VALUE is non-nil, fixed-pitch fonts conform, vice-virsa.)\n\
\n\
This function returns a list that consists of the score of the element of CANDIDATES,\n\
and the order is the same with CANDIDATES.")
     (requests, candidates)
{
  logfont_candidates *plc, *plc2;
  Lisp_Object curcand, tem, tem2, curreq;
  Lisp_Object result = Qnil;
  HFONT hf;
  int *psc;
  int *pscore;
  int num;
  int i;

  num = Flength(candidates);
  plc = (logfont_candidates*) xmalloc(sizeof(logfont_candidates) * num);
  pscore = (int *) xmalloc(sizeof(int) * num);
  memset(pscore, 0, sizeof(int) * num);

  plc2 = plc;
  for (i = 0;i < num;i++)
    {
      CHECK_LIST(candidates, 0);
      curcand = XCONS(candidates)->car;
      CHECK_LIST(curcand, 0);
      plc2->family = XCONS(curcand)->car;
      CHECK_STRING(plc2->family, 0);
      tem = XCONS(curcand)->cdr;
      CHECK_LIST(tem, 1);
      tem = XCONS(tem)->cdr;
      CHECK_LIST(tem, 1);
      tem2 = XCONS(tem)->car;
      if (EQ(tem2, Qscalable))
	plc2->type = 1;
      else
	plc2->type = 0;
      tem = XCONS(tem)->cdr;
      CHECK_LIST(tem, 0);
      tem2 = XCONS(tem)->car;
      Fw32_check_logfont(tem2);
      plc2->lf = lisp_object_to_logfont(tem2);

      plc2->width = plc2->height
	= plc2->base = plc2->overhang = plc2->charset = -1;
      hf = CreateFontIndirect(&plc2->lf);
      if (hf)
	{
	  HDC hdc;
	  HANDLE oldobj;
	  TEXTMETRIC tm;
	  HWND hwnd;
	  int flag;

	  hwnd = ROOT_WINDOW;
	  hdc = MyGetDC(hwnd);
	  oldobj = SelectObject(hdc, hf);
	  flag = GetTextMetrics(hdc, &tm);
	  SelectObject(hdc, oldobj);
	  DeleteObject(hf);
	  ReleaseDC(hwnd, hdc);
	  if (flag)
	    {
	      plc2->width = tm.tmAveCharWidth;
	      plc2->height = tm.tmHeight;
	      plc2->base = tm.tmAscent;
	      plc2->overhang = tm.tmOverhang;
	      plc2->charset = tm.tmCharSet;
	    }
	}
      plc2++;
      candidates = XCONS(candidates)->cdr;
    }

  for (;!NILP(requests);requests = XCONS(requests)->cdr)
    {
      CHECK_LIST(requests, 0);
      curreq = XCONS(requests)->car;
      plc2 = plc;
      psc = pscore;
      for (i = 0;i < num;i++)
	{
	  *psc += w32_score_logfont(plc2, curreq);
	  plc2++;
	  psc++;
	}
    }

  for (i = (num - 1);i >= 0;i--)
    result = Fcons(make_number(pscore[i]), result);
  
  return result;
}

/* syms */
syms_of_mw32font()
{
  Qw32_logfont = intern ("w32-logfont");
  staticpro (&Qw32_logfont);
  Qbdf_font = intern ("bdf-font");
  staticpro (&Qbdf_font);

  Qweight = intern ("weight");
  staticpro (&Qweight);
  Qbase = intern ("base");
  staticpro (&Qbase);
  Qfixed = intern ("fixed");
  staticpro (&Qfixed);
  Qitalic = intern ("italic");
  staticpro (&Qitalic);
  Qfamily = intern ("family");
  staticpro (&Qfamily);
  Qraster = intern ("raster");
  staticpro (&Qraster);
  Qscalable = intern ("scalable");
  staticpro (&Qscalable);

  DEFVAR_LISP ("w32-system-coding-system",
	       &Vw32_system_coding_system,
	       "coding system in windows system");
  Vw32_system_coding_system = Qnil;

  DEFVAR_LISP ("w32-default-font-alist",
	       &Vw32_default_font_alist,
	       "* When font is created, ommited parameters is completed by\n\
this alist.");
  Vw32_default_font_alist = Qnil;

  defsubr (&Sw32_check_logfont);
  defsubr (&Sw32_add_font);
  defsubr (&Sw32_change_font_logfont);
  defsubr (&Sw32_change_font_attribute);
  defsubr (&Sw32_font_list);
  defsubr (&Sw32_get_font_logfont);
  defsubr (&Sw32_get_font_info);
  defsubr (&Sw32_get_logfont_info);
  defsubr (&Sw32_query_get_logfont);
  defsubr (&Sw32_enum_logfont);
  defsubr (&Sw32_score_logfont_candidates);
/*
   defsubr(&Sw32_delete_fontset);
   defsubr(&Sw32_delete_font);
   defsubr(&Sw32_get_font_info);
*/
  /* Setting callback functions for fontset handler.  */

  get_font_info_func = w32_get_font_info;
  list_fonts_func = Fw32_font_list;
  load_font_func = w32_load_font;
  query_font_func = w32_query_font;
  set_frame_fontset_func = w32_set_font;
  find_ccl_program_func = w32_find_ccl_program;
  check_window_system_func = check_w32gui;

}
